import java.awt.*;
import java.util.Vector;

/*************
 * Il segno che rappresenta un nodo 
 * 19/12/2012
 *       aggiunta la visibilit del nodo
 * 27/5/2019
 *       impedito lo spostamento di un nodo fissato
 * @version 2
 *************/
public class NodeSign<TN>
{
/*************
 * Il nodo rappresentato
 *************/
Node<TN> nn;
/*************
 * Il colore del nodo
 *************/
Color c;
/*************
 * Il tipo di forma usata per rappresentare il nodo 
 *************/
static final int CIRCLE=0, SQUARE=1, CROSS=2;
/*************
 * Il tipo di forma usata per rappresentare il nodo 
 *************/
int shape;
/*************
 * Se il nodo e' rappresentato da una forma piena e' true, false altrimenti 
 *************/
boolean filled;
/*************
 * Le coordinate del nodo nel piano 
 *************/
int x,y;
/*************
 * Lo spostamento delle coordinate del nodo nel piano quando si adatta 
 *************/
double dx,dy;
/*************
 * Se false consente lo spostamento delle coordinate del nodo nel piano, altrimenti lo impedisce 
 *************/
boolean fixed;
/*************
 * La dimensione del segno 
 *************/
int dim;
/*************
 * L'immagine che rappresenta il segno 
 *************/
Image im;
/*************
 * Il nome dell'immagine che rappresenta il segno 
 *************/
String imn;
/*************
 * Se il nodo e' selezionato e' true, false altrimenti 
 *************/
boolean selected;
/*************
 * Lo spessore del bordo usato per rappresentare il nodo 
 *************/
int th;
/*************
 * Se il nodo e' visibile e' true, false altrimenti 
 *************/
boolean visible=true;
/*************
 * La dimensione default dei segni 
 *************/
static int nodeSize=10;
/*************
 * I listeners del nodo 
 *************/
Vector <GraphListener>graphListeners=new Vector <GraphListener>();

/*************
 * Aggiunge un nuovo GraphListener al pannello.
 * @param gl il nuovo GraphListener da aggiungere
 *************/
public void addGraphListener(GraphListener gl)
  {
  graphListeners.add(gl);
  }
/*************
 * Rimuove un GraphListener dal pannello.
 * @param gl il nuovo GraphListener da aggiungere
 *************/
public void removeGraphListener(GraphListener gl)
  {
  graphListeners.removeElement(gl);
  }
 
/*************
 * Rimuove un nuovo GraphListener dal pannello.
 * @return la matrice contenente i GraphListener aggiunti al pannello
 *************/
public GraphListener[] getGraphListeners()
  {
  GraphListener listeners[]=new GraphListener[graphListeners.size()];
  for (int i=0;i<graphListeners.size();i++)
    listeners[i]=graphListeners.elementAt(i);
  return listeners;
  }

/*************
 * Avverte tutti i listeners di un cambiamento del nodo
 *************/
public void warnChanged()
  {
  for (int i=0;i<graphListeners.size();i++)  // per attivare i listeners
    graphListeners.elementAt(i).changed(new GraphEvent(this));
  }
  
/*************
 * Modifica i valori di default.
 * @param ns dimensione del nodo di default
 *************/
public static void setDefault(int ns)
  {
  nodeSize=ns;
  }
/*************
 * Crea un nuovo nodo 
 * @param name il nome del nodo rappresentato
 * @param x l'ascissa del nodo sul piano
 * @param y l'ordinata del nodo sul piano
 *************/
public NodeSign(String name, int x, int y)
  {
  this(new Node<TN>(name),x,y,Color.black);
  }
/*************
 * Crea un nuovo nodo 
 * @param n il nodo rappresentato
 * @param x l'ascissa del nodo sul piano
 * @param y l'ordinata del nodo sul piano
 *************/
public NodeSign(Node<TN> n, int x, int y)
  {
  this(n,x,y,Color.black);
  }
/*************
 * Crea un nuovo nodo 
 * @param name il nome del nodo rappresentato
 * @param x l'ascissa del nodo sul piano
 * @param y l'ordinata del nodo sul piano
 * @param c il colore del  nodo
 *************/
public NodeSign(String name, int x, int y, Color c)
  {
  this(new Node<TN>(name),x,y,c);
  }
/*************
 * Crea un nuovo nodo 
 * @param n il nodo rappresentato
 * @param x l'ascissa del nodo sul piano
 * @param y l'ordinata del nodo sul piano
 * @param c il colore del  nodo
 *************/
public NodeSign(Node<TN> n, int x, int y, Color c)
  {
  this(n,x,y,(c!=null)?c:Color.black,nodeSize);
  }
/*************
 * Crea un nuovo nodo 
 * @param name il nome del nodo rappresentato
 * @param x l'ascissa del nodo sul piano
 * @param y l'ordinata del nodo sul piano
 * @param c il colore del  nodo
 * @param dim la dimensione del segno
 *************/
public NodeSign(String name, int x, int y, Color c, int dim)
  {
  this(new Node<TN>(name),x,y,c,dim);
  }
/*************
 * Crea un nuovo nodo 
 * @param n il nodo rappresentato
 * @param x l'ascissa del nodo sul piano
 * @param y l'ordinata del nodo sul piano
 * @param c il colore del  nodo
 * @param dim la dimensione del segno
 *************/
public NodeSign(Node<TN> n, int x, int y, Color c, int dim)
  {
  this(n,x,y,c,dim,null,null);
  }
/*************
 * Crea un nuovo nodo 
 * @param n il nodo rappresentato
 * @param x l'ascissa del nodo sul piano
 * @param y l'ordinata del nodo sul piano
 * @param c il colore del  nodo
 * @param dim la dimensione del segno
 * @param im l'immagine che rappresenta il segno
 * @param imn il nome dell'immagine che rappresenta il segno
 *************/
public NodeSign(Node<TN> n, int x, int y, Color c, int dim,Image im, String imn)
  {
  this(n,x,y,c,dim,im,imn,0 /* CIRCLE */,false);
  }
/*************
 * Crea un nuovo nodo 
 * @param n il nodo rappresentato
 * @param x l'ascissa del nodo sul piano
 * @param y l'ordinata del nodo sul piano
 * @param c il colore del  nodo
 * @param dim la dimensione del segno
 * @param shape la forma che rappresenta il nodo (CIRCLE=0, SQUARE=1, CROSS=2, poligono=4...n)
 * @param filled <b>true</b> se la forma che rappresenta il nodo  piena, <b>false</b> altrimenti 
 *************/
public NodeSign(Node<TN> n, int x, int y, Color c, int dim, int shape, boolean filled)
  {
  this(n,x,y,c,dim,null,null,shape,filled);
  }
/*************
 * Crea un nuovo nodo 
 * @param n il nodo rappresentato
 * @param x l'ascissa del nodo sul piano
 * @param y l'ordinata del nodo sul piano
 * @param c il colore del  nodo
 * @param dim la dimensione del segno
 * @param shape la forma che rappresenta il nodo (CIRCLE=0, SQUARE=1, CROSS=2, poligono=4...n)
 * @param filled <b>true</b> se la forma che rappresenta il nodo  piena, <b>false</b> altrimenti 
 * @param thickness lo spessore del bordo del segno 
 *************/
public NodeSign(Node<TN> n, int x, int y, Color c, int dim, int shape, boolean filled, int thickness)
  {
  this(n,x,y,c,dim,null,null,shape,filled,thickness);
  }
/*************
 * Crea un nuovo nodo 
 * @param n il nodo rappresentato
 * @param x l'ascissa del nodo sul piano
 * @param y l'ordinata del nodo sul piano
 * @param c il colore del  nodo
 * @param dim la dimensione del segno
 * @param im l'immagine che rappresenta il segno
 * @param imn il nome dell'immagine che rappresenta il segno
 * @param shape la forma che rappresenta il nodo (CIRCLE=0, SQUARE=1, CROSS=2, poligono=4...n)
 * @param filled <b>true</b> se la forma che rappresenta il nodo  piena, <b>false</b> altrimenti 
 *************/
public NodeSign(Node<TN> n, int x, int y, Color c, int dim,Image im, String imn, int shape, boolean filled)
  {
  this(n,x,y,c,dim,im,imn,shape,filled,0);
  }
/*************
 * Crea un nuovo nodo 
 * @param n il nodo rappresentato
 * @param x l'ascissa del nodo sul piano
 * @param y l'ordinata del nodo sul piano
 * @param c il colore del  nodo
 * @param dim la dimensione del segno
 * @param im l'immagine che rappresenta il segno
 * @param imn il nome dell'immagine che rappresenta il segno
 * @param shape la forma che rappresenta il nodo (CIRCLE=0, SQUARE=1, CROSS=2, poligono=4...n)
 * @param filled <b>true</b> se la forma che rappresenta il nodo  piena, <b>false</b> altrimenti 
 * @param thickness lo spessore del bordo del segno 
 *************/
public NodeSign(Node<TN> n, int x, int y, Color c, int dim,Image im, String imn, int shape, boolean filled, int thickness)
  {
  this(n,x,y,c,dim,im,imn,shape,filled,0,false);
  }
/*************
 * Crea un nuovo nodo 
 * @param n il nodo rappresentato
 * @param x l'ascissa del nodo sul piano
 * @param y l'ordinata del nodo sul piano
 * @param c il colore del  nodo
 * @param dim la dimensione del segno
 * @param im l'immagine che rappresenta il segno
 * @param imn il nome dell'immagine che rappresenta il segno
 * @param shape la forma che rappresenta il nodo (CIRCLE=0, SQUARE=1, CROSS=2, poligono=4...n)
 * @param filled <b>true</b> se la forma che rappresenta il nodo  piena, <b>false</b> altrimenti 
 * @param thickness lo spessore del bordo del segno 
 * @param fixed vero se il segno  fissato 
 *************/
public NodeSign(Node<TN> n, int x, int y, Color c, int dim,Image im, String imn, int shape, boolean filled, int thickness, boolean fixed)
  {
  nn=n;
/*
  if (x<0 || y<0)
    {
    int xn=size/2,yn=size/2;
    this.x=xn;
    this.y=yn;
    }
   else
*/
    {
    this.x=x;
    this.y=y;
    }
  this.c=c;
  this.dim=dim;
  this.im=im;
  this.imn=imn;
  this.shape=shape;
  this.filled=filled;
  this.th=thickness;
  this.fixed=fixed;
  selected=false;
  }
/*************
 * Cambia il nodo rappresentato
 * @param n il nuovo nodo rappresentato
 */
public void setNode(Node<TN> n)
  {
  if (!n.equals(nn))
    warnChanged();
  nn=n;
  }
/*************
 * Ritorna il nodo rappresentato
 * @return il nodo rappresentato
 */
public Node<TN> getNode()
  {
  return nn;
  }
/*************
 * Ritorna il nome del nodo rappresentato
 * @return il nome del nodo rappresentato
 */
public String getName()
  {
  return getNode().getName();
  }
/*************
 * Cambia l'ascissa del nodo
 * @param nx la nuova ascissa del nodo
 */
public void setX(int nx)
  {
  if (!fixed)
    {
    if (x!=nx)
      warnChanged();
    x=nx;
    }
  }
/*************
 * Ritorna l'ascissa del nodo
 * @return l'ascissa del nodo
 */
public int getX()
  {
  return x;
  }
/*************
 * Aggiunge il valore allo spostamento dell'ascissa del nodo
 * @param dx la modifica dello spostamento dell'ascissa del nodo
 */
public void addDx(double dx)
  {
  if (this.dx!=this.dx+dx)
    warnChanged();
  this.dx+=dx;
  }
/*************
 * Ritorna il valore allo spostamento dell'ascissa del nodo
 * @return il valore allo spostamento dell'ascissa del nodo
 */
public double getDx()
  {
  return this.dx;
  }
/*************
 * Modifica il valore dello spostamento dell'ascissa del nodo
 * @param dx la modifica dello spostamento dell'ascissa del nodo
 */
public void setDx(double dx)
  {
  if (this.dx!=dx)
    warnChanged();
  this.dx=dx;
  }
/*************
 * Cambia l'ordinata del nodo
 * @param ny la nuova ordinata del nodo
 */
public void setY(int ny)
  {
  if (!fixed)
    {
    if (y!=ny)
      warnChanged();
    y=ny;
    }
  }
/*************
 * Ritorna l'ordinata del nodo
 * @return l'ordinata del nodo
 */
public int getY()
  {
  return y;
  }
/*************
 * Aggiunge il valore allo spostamento dell'ordinata del nodo
 * @param dy la modifica dello spostamento dell'ordinata del nodo
 */
public void addDy(double dy)
  {
  if (this.dy!=this.dy+dy)
    warnChanged();
  this.dy+=dy;
  }
/*************
 * Ritorna il valore dello spostamento dell'ordinata del nodo
 * @return il valore dello spostamento dell'ordinata del nodo
 */
public double getDy()
  {
  return this.dy;
  }
/*************
 * Modifica il valore allo spostamento dell'ordinata del nodo
 * @param dy la modifica dello spostamento dell'ordinata del nodo
 */
public void setDy(double dy)
  {
  if (this.dy!=dy)
    warnChanged();
  this.dy=dy;
  }
/*************
 * Stabilisce se il nodo  muovibile
 * @param f <b>true</b> se il nodo  muovibile, <b>false</b> altrimenti
 */
public void setFixed(boolean f)
  {
  if (fixed!=f)
    warnChanged();
  fixed=f;
  }
/*************
 * Ritorna la movibilita' del nodo
 * @return <b>true</b> se il nodo  movibile, <b>false</b> altrimenti
 */
public boolean isFixed()
  {
  return fixed;
  }
/*************
 * Cambia il nuovo colore del nodo
 * @param col il nuovo colore del nodo
 */
public void setColor(Color col)
  {
  if (!c.equals(col))
    warnChanged();
  c=col;
  }
/*************
 * Ritorna il colore del nodo
 * @return il colore del nodo
 */
public Color getColor()
  {
  return c;
  }
/*************
 * Cambia la nuova dimensione del nodo
 * @param size la nuova dimensione del nodo
 */
public void setSize(int size)
  {
  if (dim!=size)
    warnChanged();
  dim=size;
  }
/*************
 * Ritorna la dimensione del nodo
 * @return la dimensione del nodo
 */
public int getSize()
  {
  return dim;
  }
/*************
 * Cambia la nuova forma del nodo
 * @param s la nuova forma del nodo, 0 cerchio, 1 quadrato, 2 croce, da 3 in poi poligono con s numero di lati
 */
public void setShape(int s)
  {
  if (shape!=s)
    warnChanged();
  shape=s<0?-s:s;
  }
/*************
 * Ritorna la dimensione del nodo
 * @return la dimensione del nodo
 */
public int getShape()
  {
  return shape;
  }
/*************
 * Modifica la campitura del nodo
 * @param f <b>true</b> se il nodo  pieno, <b>false</b> altrimenti
 */
public void setFilled(boolean f)
  {
  if (filled!=f)
    warnChanged();
  filled=f;
  }
/*************
 * Ritorna la campitura del nodo
 * @return <b>true</b> se il nodo  campito, <b>false</b> altrimenti
 */
public boolean isFilled()
  {
  return filled;
  }
/*************
 * Cambia la nuova immagine del nodo
 * @param img la nuova immagine del nodo
 */
public void setImage(Image img)
  {
  warnChanged();
  setImage(null,img);
  }
/*************
 * Cambia la nuova immagine del nodo
 * @param imgn il nome della nuova immagine del nodo
 * @param img la nuova immagine del nodo
 */
public void setImage(String imgn, Image img)
  {
  if (imgn!=null)
    imn=imgn.replace(' ','_');
  im=img;
  if (im==null)
    {
    imn=null;
    }
  }
/*************
 * Ritorna il nome della immagine del nodo
 * @return il nome della immagine del nodo
 */
public String getImageName()
  {
  return imn;
  }
/*************
 * Ritorna l'immagine del nodo
 * @return l'immagine del nodo
 */
public Image getImage()
  {
  return im;
  }
/*************
 * Ritorna lo spessore del bordo del nodo
 * @return lo spessore del bordo del nodo
 *************/
public int getThickness()
  {
  return th;
  }
/*************
 * Cambia lo spessore del bordo del nodo
 * @param thick il nuovo spessore del bordo del nodo
 *************/
public void setThickness(int thick)
  {
  if (thick!=th)
    warnChanged();
  th=thick;
  }
/*************
 * Modifica la visibilita' del nodo
 * @param vis <b>true</b> se il nodo  visibile, <b>false</b> altrimenti
 */
public void setVisible(boolean vis)
  {
  if (vis!=visible)
    warnChanged();
  visible=vis;
  }
/*************
 * Ritorna la visibilita' del nodo
 * @return <b>true</b> se il nodo  visibile, <b>false</b> altrimenti
 */
public boolean isVisible()
  {
  return visible;
  }
/*************
* Disegna il GraphPanel 
* @param g lo spazio grafico di tracciamento
* @param dx l'offset dell'immagine
* @param dy l'offset dell'immagine
* @param z lo zoom dell'immagine
*************/
public void paint(Graphics g,int dx,int dy,double z)
  {
  paint(g,dx,dy,z,true);
  }
/*************
 * Disegna il GraphPanel 
 * @param g lo spazio grafico di tracciamento
 * @param dx l'offset dell'immagine
 * @param dy l'offset dell'immagine
 * @param z lo zoom dell'immagine
 * @param showtext true se va visualizzato il testo del nome
 *************/
public void paint(Graphics g,int dx,int dy,double z,boolean showtext)
  {
  int x0=(int)((dx+x)*z);
  int y0=(int)((dy+y)*z);
  g.setColor(c);
  int nodeSize=(int)(dim*z);
  if (selected)
      if (shape==CIRCLE)
        g.drawOval(x0-nodeSize-1,y0-nodeSize-1,2*nodeSize+2,2*nodeSize+2);
       else if (shape==SQUARE)
        g.drawRect(x0-nodeSize-1,y0-nodeSize-1,2*nodeSize+2,2*nodeSize+2);
       else if (shape==CROSS)
        {
        g.drawRect(x0-nodeSize/10-1,y0-nodeSize-1,nodeSize/5+2,2*nodeSize+2);
        g.drawRect(x0-nodeSize-1,y0-nodeSize/10-1,2*nodeSize+2,nodeSize/5+2);
        }
       else
        {
        int px[]=new int[shape];
        int py[]=new int[shape];
        double drad=(2*Math.PI/shape);
//        double rad=0;
        double rad=-Math.PI/2;
        for (int liv=0;liv<shape;liv++,rad+=drad)
          {
          int radius= nodeSize+1;
          px[liv]=x0+(int)(Math.cos(rad)*radius);
          py[liv]=y0+(int)(Math.sin(rad)*radius);
    //      System.out.println("spigolo "+liv+" "+rad+" "+radius+" "+px[liv]+" "+py[liv]);
          }
        g.drawPolygon(px,py,shape);
        }
  if (visible)
    {
    if (im!=null)
      g.drawImage(im,x0-nodeSize,y0-nodeSize,2*nodeSize,2*nodeSize,null);
     else
      if (filled)
        {
        if (shape==CIRCLE)
          g.fillOval(x0-nodeSize,y0-nodeSize,2*nodeSize,2*nodeSize);
         else if (shape==SQUARE)
          g.fillRect(x0-nodeSize,y0-nodeSize,2*nodeSize,2*nodeSize);
         else if (shape==CROSS)
          {
          g.fillRect(x0-nodeSize/10,y0-nodeSize,nodeSize/5,2*nodeSize);
          g.fillRect(x0-nodeSize,y0-nodeSize/10,2*nodeSize,nodeSize/5);
          }
         else
          {
          int px[]=new int[shape];
          int py[]=new int[shape];
          double drad=(2*Math.PI/shape);
  //        double rad=0;
          double rad=-Math.PI/2;
          for (int liv=0;liv<shape;liv++,rad+=drad)
            {
            int radius= nodeSize;
            px[liv]=x0+(int)(Math.cos(rad)*radius);
            py[liv]=y0+(int)(Math.sin(rad)*radius);
      //      System.out.println("spigolo "+liv+" "+rad+" "+radius+" "+px[liv]+" "+py[liv]);
            }
          g.fillPolygon(px,py,shape);
          }
        }
       else
        {
        if (shape==CIRCLE)
          {
          g.drawOval(x0-nodeSize,y0-nodeSize,2*nodeSize,2*nodeSize);
          for (int thk=1;thk<th;thk++)
            {
            g.drawOval(x0-nodeSize+thk,y0-nodeSize+thk,2*nodeSize-2*thk,2*nodeSize-2*thk);
            }
          }
         else if (shape==SQUARE)
          {
          g.drawRect(x0-nodeSize,y0-nodeSize,2*nodeSize,2*nodeSize);
          for (int thk=1;thk<th;thk++)
            {
            g.drawRect(x0-nodeSize+thk,y0-nodeSize+thk,2*nodeSize-2*thk,2*nodeSize-2*thk);
            }
          }
         else if (shape==CROSS)
          {
          g.drawRect(x0-nodeSize/10,y0-nodeSize,nodeSize/5,2*nodeSize);
          g.drawRect(x0-nodeSize,y0-nodeSize/10,2*nodeSize,nodeSize/5);
          for (int thk=1;thk<th;thk++)
            {
            g.drawRect(x0-nodeSize/10+thk,y0-nodeSize+thk,nodeSize/5-2*thk,2*nodeSize-2*thk);
            g.drawRect(x0-nodeSize+thk,y0-nodeSize/10+thk,2*nodeSize-2*thk,nodeSize/5-2*thk);
            }
          }
         else
          {
          int px[]=new int[shape];
          int py[]=new int[shape];
          double drad=(2*Math.PI/shape);
  //        double rad=0;
          double rad=-Math.PI/2;
          for (int liv=0;liv<shape;liv++,rad+=drad)
            {
            int radius= nodeSize;
            px[liv]=x0+(int)(Math.cos(rad)*radius);
            py[liv]=y0+(int)(Math.sin(rad)*radius);
      //      System.out.println("spigolo "+liv+" "+rad+" "+radius+" "+px[liv]+" "+py[liv]);
            }
          g.drawPolygon(px,py,shape);
          for (int thk=1;thk<th;thk++)
            {
            rad=-Math.PI/2;
            for (int liv=0;liv<shape;liv++,rad+=drad)
              {
              int radius= nodeSize-thk;
              px[liv]=x0+(int)(Math.cos(rad)*radius);
              py[liv]=y0+(int)(Math.sin(rad)*radius);
              }
            g.drawPolygon(px,py,shape);
            }
          }
        }
    }
  if (showtext)
    {
    if (filled && shape!=CROSS)
      {
      int ac=c.getRGB();
      int lum=((ac&0xff0000)>>16)|((ac&0x00ff00)>>8)|(ac&0x0000ff);
      if ((lum&0xc0)!=0xc0)
        g.setColor(Color.white);
       else
        g.setColor(Color.black);
      }
    g.drawString(nn.name,x0-g.getFontMetrics().stringWidth(nn.name)/2,y0);
    }
  }
/************
 * Ritorna la stringa che descrive il nodo
 * @return la descrizione del nodo
 ************/
public String toString()
  {
  return "NodeSign["+nn+",("+x+","+y+"),"+dim+","+c+","+shape+","+filled+","+th+","+visible+"]";
  }
}
