import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import javax.swing.*;
import java.util.*;
import java.io.*;
import java.net.*;
import java.security.*;
import java.lang.reflect.*;

/*****
 * GraphJPanel e' un JPanel su cui e' visualizzato un grafo 
 * @version 4.5
 * 
 * 1/3/2011
 *      corretto rappresentazione degli anelli
 *      corretto collocazione dei nodi di una matrice
 * 19/12/2012
 *      reso GraphJPanel un ImageReader
 *      aggiunto la possibilità di indicare i colori da file di testo in altre basi
 *      sistemato il flickering
 * 20/12/2012
 *      salva il nome del background
 * 29/12/2012
 *      gestione del colore dei singoli archi (non ancora visualizzato autonomamente)
 * 1/3/2013
 *      aggiunto costruttore che fornisce la collocazione dei nodi
 * 4/3/2013
 *      aggiunto metodi che ritornano il nodo dato l'indice e l'arco dati i nodi
 *      aggiunto la possibilità di modificare i testi dei menu.
 * 3/12/2013
 *		  aggiunto metodi per la rappresentazione di un nodo con diverse forme e con riempimento
 * 6/5/2014
 *      aggiunto metodi per abilitare e disabilitare le modifiche, lo scivolamento e zoom interattivi
 *      sistemato la lettura di immagini da applet
 * 23/5/14
 *      aggiunte le nuove forme dei nodi
 *      aggiunta la gestione di GraphListener
 * 8/7/14
 *      aggiunta la visibilità dei percorsi
 *      aggiunta lo spessore dei segni
 *      aggiunta visualizzazione separata del colore dei pesi negli archi
 *      aggiunta la selezione di piu' nodi per modificarli assieme (muoverli e cambiarli) (SHIFT in spazio aperto per rettangolo di selezione)
 * 23/12/15
 *      aggiunta la selezione multipla incrementale un nodo alla volta. (selezione del nodo con SHIFT)
 *      aggiunta la possibilità di non deselezionare la selezione multipla tenendo lo SHIFT abbassato quando si attiva il menù contestuale
 * 31/8/17
 *      abilitati i listeners quando il popup è disabilitato
 * 14/4/18
 *      abilitati i listeners quando editing disabilitato
 *      aggiunto i listeners ad archi e nodi e la relativa sensibilità ai cambiamenti del GraphJPanel
 *      aggiunta la distinzione tra editabilita' e muovibilita'
 * 15/4/18
 *      modificata l'interfaccia GraphListener aggiungendo la cancellazione e l'inserimento di nodi e archi
 * 16/4/18
 *      corretta la selezione multipla con rettangoli selezionati nelle diverse direzioni
 * 17/4/18
 *      corretto un errore di visualizzazione degli archi perfettamente verticali verso e da nodi quadrati
 * 23/4/18
 *      aggiunto il comportamento adattivo e la possibilità di fissare i nodi nell'adattamento e negli spostamenti multipli
 * 16/1/19
 *      corretto il nome del metodo da getadjacency a getAdjacency
 * 21/1/19
 *      sostituito BUTTON1_MASK con BUTTON1_DOWN_MASK , BUTTON3_MASK con BUTTON3_DOWN_MASK e getModifiers con getModifiersEx
 * 12/2/19
 *      sistemato il default di fissaggio dei nodi
 * 27/5/19
 *      bloccato lo spostamento dei nodi fissati anche non durante l'adattamento
 *      sistemato il cambio di caratteristica con le selezioni multiple.
 * 6/2/20
 *      aggiustato la segnalazione di archi vuoti in seguito a cancellazione di nodo
 *****/
public class GraphJPanel<TN,TA> extends JPanel implements MouseListener, MouseMotionListener, ActionListener, ImageReader, GraphListener
{
static final long serialVersionUID=1;
NodeSign<TN> n[];
ArcSign<TN,TA> a[][];
Graph<TN,TA> g;
Color defaultColor=Color.black;
double defaultWeight=1.0;
String defaultName="n";
int defaultSize=10;
int defaultShape=0;
boolean defaultFill=false;
boolean defaultFixing=false;
String defaultImage=null;
int defaultThickness=0;
int zoomRate=0;
boolean showAxis=false;
boolean showNames=true;
boolean showWeights=true;
boolean adaptive=false;
Thread relaxer=null;
final static int MINZOOM=-2,MAXZOOM=7;
final static int MFILE=0,MEDIT=1,MSERACH=2,MCOL=3,MZOOM=4;
final static int PWEIGHT=1,PREN1=2,PNAME=3,PZOOM=4,PCOLOR=5,PSIZE=6,PIMAGE=7,PSHAPE=8,PFILL=9,PTHICKN=10,PTHICKA=11,/*senza scritta */PMINL1=12,PMINP1=13,PREN=14,PMINL=15,PMINP=16,PTHICK=17;
//final static int SCIRCLE=12,SSQUARE=SCIRCLE+1,SCROSS=SCIRCLE+2,SNO=SCIRCLE+3,SYES=SCIRCLE+4,SFALSE=SCIRCLE+5,STRUE=SCIRCLE+6,SFOR=SCIRCLE+7;
final static int SCIRCLE=12,SSQUARE=13,SCROSS=14,SNO=15,SYES=16,SFALSE=17,STRUE=18,SFOR=19,/*con scritta*/PFIX=20,PINFO=21;
final static int VDELN=1,VDELA=2,VMINP=3,VMINL=4,VNAME=5,VCOL=6,VWEI=7,VZOOM=8,VSIZE=9,VIMAGE=10,VSHAPE=11,VFILL=12,VTHICK=13,VFIX=14,VINFO=15;
String voice[]={"",
                "Delete node",
                "Delete arc",
                "Min price -> other node",
                "Min length -> other node",
                "Change name",  //5
                "Change color",
                "Change weight",
                "Change zoom",
                "Change size",
                "Change image",  //10
                "Change shape",
                "Change filling",
                "Change thickness",
                "Change fixing",
                "Get info",       //15
               };
String prompt[]={"",
                 "weight of arcs:",
                 "new name for ",
                 "node name:",
                 "zoom (-2..7):",
                 "choose a color", //5
                 "new size",
                 "new image",
                 "new shape",
                 "set filling",
                 "new thickness",  //10
                 "new thickness of the arc:",
                 "circle",
                 "square",
                 "cross",
                 "no", //15
                 "yes",
                 "false",
                 "true",
                 "for",
                 "set fixing", //20
                 "node information",
                };
Image bgImage=null;
String bgName;
ImageReader ir=this;
boolean modified=false;
final static int prefSize=500;
int size=prefSize;
int operation=0;
boolean usePopUp=true, movable=true, editing=true, sliding=true;
JPopupMenu mnPointPU;
JMenuItem mnRemPU,mnUnlinkPU,mnMinLPU,mnMinPPU,mnRenPU,mnColSetPU,mnWeightPU,mnZoomPU,mnSizePU,mnImagePU,mnShapePU,mnFillPU,mnThickPU,mnFixPU,mnInfoPU;
MouseEvent lastEvent;
@SuppressWarnings({"rawtypes","unchecked"})
Info info=null;
boolean dropSelected=false;
String first;
Vector <GraphListener>graphListeners=new Vector <GraphListener>();

/******
 * Crea un GraphJPanel di un grafo vuoto
 *******/
public GraphJPanel()
  {
  this(new Graph<TN,TA>());
  }
/******
 * Crea un GraphJPanel di un grafo vuoto con una immagine di sfondo
 * @param bg l'immagine di sfondo
 * @param bgn il nome dell'immagine di sfondo
 *******/
public GraphJPanel(Image bg, String bgn)
  {
  this(new Graph<TN,TA>(),bg,bgn);
  }
/******
 * Crea un GraphJPanel di un grafo dato
 * @param gr il grafo
 *******/
public GraphJPanel(Graph<TN,TA> gr)
  {
  this(gr,(Image)null,(String)null);
  }
/******
 * Crea un GraphJPanel di un grafo dato con una immagine di sfondo
 * @param gr il grafo
 * @param bg l'immagine di sfondo
 * @param bgn il nome dell'immagine di sfondo
 *******/
public GraphJPanel(Graph<TN,TA> gr, Image bg, String bgn)
  {
  g=gr;
  bgImage=bg;
  bgName=bgn;
  setGraph(g);
  setListeners();
  }
/******
 * Crea un GraphJPanel prelevando dal Reader r la descrizione di un grafo in due formati. Nel primo il formato
 * prevede che la prima riga letta contenga il numero di nodi e la successiva riga contiene l'elenco dei nomi dei nodi
 * e nelle righe successive sono conservati i valori della matrice di adiacenza
 * dove le colonne e le righe conservano lo stesso ordine dei nomi dei nodi
 * Successivamente puo' esserci la matrice dei pesi
 * dove le colonne e le righe conservano lo stesso ordine dei nomi dei nodi
 * e gli elementi sono i valori dei singoli archi separati da '|'
 * In seguito, se c'e' un'altra riga, contiene una terna  di valori separati da virgola per ogni nodo
 * contenente il  valore del codice del colore in decimale e le coordinate x e y del nodo nel grafo.
 * Infine dopo puo' esserci una matrice contenente i valori dei colori degli archi.
 * Nel secondo formato invece sono elencati su ogni riga la coppia di nodi connessi da un arco ed eventualmente il costo dell'arco.
 * Il nome del primo nodo non puo' essere un numero intero
 * @param r il Reader da cui viene prelevato il grafo
 *******/
public GraphJPanel (Reader r)
  {
  load(r);
  setListeners();
  }
/******
 * Crea un GraphJPanel contente un grafo di cui viene fornita la matrice di adiacenza 
 * e la matrice contente i nomi dei nodi
 * @param name nomi dei nodi del grafo
 * @param matAd matrice di adiacenza del grafo
 ******/
public GraphJPanel(String name[], int matAd[][])
  {
  this(name,matAd,null);
  }
/******
 * Crea un GraphJPanel contenente un grafo di cui viene fornita la matrice di adiacenza,
 * la matrice contente i nomi dei nodi e
 * la matrice dei prezzi di ogni arco.
 * @param name nomi dei nodi del grafo
 * @param matAd matrice di adiacenza del grafo
 * @param p matrice dei prezzi di ogni arco
 *******/
public GraphJPanel(String name[], int matAd[][], double p[][][])
  {
  this(name,null,matAd,p);
  }
/******
 * Crea un GraphJPanel contenente un grafo di cui viene fornita la matrice di adiacenza,
 * la matrice contente i nomi dei nodi e
 * la matrice dei prezzi di ogni arco.
 * @param name nomi dei nodi del grafo
 * @param pos le coordinate dei nodi (primo indice orizzontale, secondo indice verticale)
 * @param matAd matrice di adiacenza del grafo
 * @param p matrice dei prezzi di ogni arco
 *******/
public GraphJPanel(String name[], int pos[][], int matAd[][], double p[][][])
  {
  this(name,pos,matAd,p,Color.black);
  }
/******
 * Crea un GraphJPanel contenente un grafo di cui viene fornita la matrice di adiacenza,
 * la matrice contente i nomi dei nodi e la matrice dei prezzi di ogni arco.
 * @param name nomi dei nodi del grafo
 * @param pos le coordinate dei nodi (primo indice orizzontale, secondo indice verticale)
 * @param matAd matrice di adiacenza del grafo
 * @param col il colore degli archi e dei nodi
 * @param p matrice dei prezzi di ogni arco
 *******/
public GraphJPanel(String name[], int pos[][], int matAd[][], double p[][][], Color col)
  {
  setGraph(new Graph<TN,TA>(name,matAd,p),pos,col);
  setListeners();
  }
/******
 * Crea un GraphJPanel contenente un grafo di cui viene fornita la lista di coppie di nodi con i relativi costi,
 * @param couples le coppie di nomi dei nodi del grafo collegati da un arco
 * @param p matrice dei prezzi di ogni arco
 *******/
public GraphJPanel(String couples[][], double p[])
  {
  this(couples,p,null,Color.black);
  }
/******
 * Crea un GraphJPanel contenente un grafo di cui viene fornita la lista di coppie di nodi con i relativi costi,
 * @param couples le coppie di nomi dei nodi del grafo collegati da un arco
 * @param p matrice dei prezzi di ogni arco
 * @param col il colore degli archi e dei nodi
 *******/
public GraphJPanel(String couples[][], double p[], Color col)
  {
  this(couples,p,null,col);
  }
/******
 * Crea un GraphJPanel contenente un grafo di cui viene fornita la lista di coppie di nodi con i relativi costi,
 * @param couples le coppie di nomi dei nodi del grafo collegati da un arco
 * @param p matrice dei prezzi di ogni arco
 * @param pos le coordinate dei nodi (primo indice orizzontale, secondo indice verticale)
 * @param col il colore degli archi e dei nodi
 *******/
public GraphJPanel(String couples[][], double p[], int pos[][], Color col)
  {
  setGraph(new Graph<TN,TA>(couples,p),pos,col);
  setListeners();
  }
/**
 * invocata quando un arco viene selezionato
 * @param ge il descrittore dell'evento
 **/
public void arcSelected(GraphEvent ge)
  {
  }
/**
 * invocata quando un nodo viene selezionato
 * @param ge il descrittore dell'evento
 **/
public void nodeSelected(GraphEvent ge)
  {
  }
/***************
 * invocata quando un arco viene aggiunto
 * @param ge il descrittore dell'evento
 ***************/
public void arcAdded(GraphEvent ge)
  {
  repaint();
  }
/***************
 * invocata quando un nodo viene aggiunto
 * @param ge il descrittore dell'evento
 ***************/
public void nodeAdded(GraphEvent ge)
  {
  repaint();
  }
/***************
 * invocata quando un arco viene rimosso
 * @param ge il descrittore dell'evento
 ***************/
public void arcRemoved(GraphEvent ge)
  {
  repaint();
  }
/***************
 * invocata quando un nodo viene rimosso
 * @param ge il descrittore dell'evento
 ***************/
public void nodeRemoved(GraphEvent ge)
  {
  repaint();
  }
/**
 * invocata quando il pannello viene modificato
 * @param ge il descrittore dell'evento
 **/
public void changed(GraphEvent ge)
  {
  repaint();
  }
/**
 * invocata quando il pannello viene spostato
 * @param ge il descrittore dell'evento
 **/
public void moved(GraphEvent ge)
  {
  repaint();
  }
/**
 * invocata quando vengono modificati i default del pannello
 * @param ge il descrittore dell'evento
 **/
public void set(GraphEvent ge)
  {
  repaint();
  }
/**
 * invocata quando viene fatto uno zoom sul pannello
 * @param ge il descrittore dell'evento
 **/
public void zoomed(GraphEvent ge)
  {
  repaint();
  }

/******
 * Predispone i listeners necessari
 ******/
void setListeners()
  {
  mnPointPU=new JPopupMenu("Point");
  mnRemPU=new JMenuItem(voice[VDELN]);
  mnUnlinkPU=new JMenuItem(voice[VDELA]);
  mnMinPPU=new JMenuItem(voice[VMINP]);
  mnMinLPU=new JMenuItem(voice[VMINL]);
  mnRenPU=new JMenuItem(voice[VNAME]);
  mnColSetPU=new JMenuItem(voice[VCOL]);
  mnWeightPU=new JMenuItem(voice[VWEI]);
  mnZoomPU=new JMenuItem(voice[VZOOM]);
  mnSizePU=new JMenuItem(voice[VSIZE]);
  mnImagePU=new JMenuItem(voice[VIMAGE]);
  mnShapePU=new JMenuItem(voice[VSHAPE]);
  mnFillPU=new JMenuItem(voice[VFILL]);
  mnThickPU=new JMenuItem(voice[VTHICK]);
  mnFixPU=new JMenuItem(voice[VFIX]);
  mnInfoPU=new JMenuItem(voice[VINFO]);
  JMenuItem mi[]={mnRemPU,mnUnlinkPU,mnMinLPU,mnMinPPU,mnRenPU,mnColSetPU,mnWeightPU,mnZoomPU,mnSizePU,mnImagePU,mnShapePU,mnFillPU,mnThickPU,mnFixPU,mnInfoPU};
  for (int i=0;i<mi.length;i++)
    {
    mi[i].addActionListener(this);
    mnPointPU.add(mi[i]);
    }
  add(mnPointPU);
  addMouseListener(this);
  addMouseMotionListener(this);
  }
/******
 * Da chiamare al momento della chiusura del JPanel
 * per rilasciare le finestre aperte e chiudere i processi in corso.
 ******/
public void close()
  {
  if (info!=null)
    {
    info.dispose();
    info=null;
    }
  adaptive=false;
  }
/******
 * Sostituisce i testi nei menu e nelle finestre
 * @param v le voci del menu
 * @param p le frasi di prompt
 ******/
public void setLanguage(String v[],String p[])
  {
//  System.out.println("Language "+v.length+" "+voice.length);
  if (v!=null)
    {
    for (int i=0;i<voice.length && i<v.length;i++)
      {
      voice[i]=v[i];
//      System.out.println("v"+i+" "+voice[i]+" "+v[i]);
      }
    mnRemPU.setText(voice[VDELN]);
    mnUnlinkPU.setText(voice[VDELA]);
    mnMinPPU.setText(voice[VMINP]);
    mnMinLPU.setText(voice[VMINL]);
    mnRenPU.setText(voice[VNAME]);
    mnColSetPU.setText(voice[VCOL]);
    mnWeightPU.setText(voice[VWEI]);
    mnZoomPU.setText(voice[VZOOM]);
    mnSizePU.setText(voice[VSIZE]);
    mnImagePU.setText(voice[VIMAGE]);
    mnShapePU.setText(voice[VSHAPE]);
    mnFillPU.setText(voice[VFILL]);
    mnThickPU.setText(voice[VTHICK]);
    mnFixPU.setText(voice[VFIX]);
    mnInfoPU.setText(voice[VINFO]);
    }
  if (p!=null)
    for (int i=0;i<prompt.length && i<p.length;i++)
      {
      prompt[i]=p[i];
//      System.out.println("p"+i+" "+prompt[i]+" "+p[i]);
      }
  }
/******
 * Abilita e disabilita scivolamento sul GraphPanel
 * @param sl <b>true</b> se si vuole abilitare lo scivolamento interattivo e <b>false</b> altrimenti
 ******/
public void setSliding(boolean sl)
  {
  sliding=sl;
  for (int i=0;i<graphListeners.size();i++)   // per attivare i listeners
    graphListeners.elementAt(i).set(new GraphEvent(this));
  }
/******
 * Controlla se e' abilitato lo scivolamento sul GraphJPanel
 * @return <b>true</b> se e' abilitato lo scivolamento interattivo, <b>false</b> altrimenti
 ******/
public boolean isSliding()
  {
  return sliding;
  }
/******
 * Abilita e disabilita gli spostamenti sul pannello
 * @param is <b>true</b> se si vuole abilitare gli spostamenti interattivi e <b>false</b> altrimenti
 ******/
public void setMovable(boolean is)
  {
  movable=is;
  for (int i=0;i<graphListeners.size();i++)   // per attivare i listeners
    graphListeners.elementAt(i).set(new GraphEvent(this));
  }
/******
 * Controlla se sono abilitati gli spostamenti sul pannello
 * @return <b>true</b> se sono abilitati gli spostamenti, <b>false</b> altrimenti
 ******/
public boolean isMovable()
  {
  return movable;
  }
/******
 * Abilita e disabilita le modifiche sul pannello
 * @param is <b>true</b> se si vuole abilitare le modifiche interattive e <b>false</b> altrimenti
 ******/
public void setEdit(boolean is)
  {
  editing=is;
  for (int i=0;i<graphListeners.size();i++)   // per attivare i listeners
    graphListeners.elementAt(i).set(new GraphEvent(this));
  }
/******
 * Controlla se sono abilitate le modifiche sul pannello
 * @return <b>true</b> se sono abilitate le modifiche interattive, <b>false</b> altrimenti
 ******/
public boolean isEdit()
  {
  return editing;
  }
/******
 * Abilita e disabilita il popup sul pannello
 * @param pu <b>true</b> se si vuole abilitare il popup e <b>false</b> altrimenti
 ******/
public void setPopUp(boolean pu)
  {
  usePopUp=pu;
  for (int i=0;i<graphListeners.size();i++)   // per attivare i listeners
    graphListeners.elementAt(i).set(new GraphEvent(this));
  }
/******
 * Controlla se e' abilitato il popup sul pannello
 * @return <b>true</b> se il popup e' abilitato, <b>false</b> altrimenti
 ******/
public boolean isPopUp()
  {
  return usePopUp;
  }
/***********
 * Notifica ai listeners che e' stato modificato il setting del pannello
  **********/
void notifySetting()
  {
  for (int i=0;i<graphListeners.size();i++)   // per attivare i listeners
    graphListeners.elementAt(i).set(new GraphEvent(this));
  }

/******
 * Segna come modificato il pannello
 * @param mo <b>true</b> se si vuole segnare come modificato il GraphPanel  e <b>false</b> altrimenti
 ******/
public void setModified(boolean mo)
  {
//  System.out.println("modified="+mo);
  modified=mo;
  if (modified)
    for (int i=0;i<graphListeners.size();i++)   // per attivare i listeners
      graphListeners.elementAt(i).changed(new GraphEvent(this));
  }
/******
 * Controlla se il pannello e' stato modificato
 * @return <b>true</b> se il popup e' modificato, <b>false</b> altrimenti
 ******/
public boolean isModified()
  {
  return modified;
  }
/******
 * Ritorna la matrice di adiacenza dei collegamenti tra i nodi in un passo
 * @return la matrice dei collegamenti in un passo
 *******/
public double[][] getAdjacency()
  {
  return g.adjacency().getMatrix();
  }
/******
 * Ritorna la matrice dei collegamenti tra i nodi in un dato numero di passi
 * @param n il numero di passi
 * @return la matrice dei collegamenti in n passi
 *******/
public double[][] getConnection(int n)
  {
  return g.adjacency().pow(n).getMatrix();
  }
/******
 * Ritorna la matrice delle connessioni tra i nodi in un numero qualsiasi di  passi
 * @return la matrice dei collegamenti in un numero qualsiasi di passi inferiori al numero dei nodi
 *******/
public double[][] getConnections()
  {
  return g.connections().getMatrix();
  }
/******
 * Linearizza il grafo (Bisogna ancora risolvere la linearizzazione di
 * piu' di due passi)
 * @return <strong>true</strong> se il grafo viene modificato, <strong>false</strong> altrimenti
 *******/
public boolean linearise()
  {
  if (g.linearise())
    {
    setModified(true);
    for (int i=0;i<g.degree();i++)
      for (int j=0;j<g.degree();j++)
        if (g.a[i][j]==null)
          a[i][j]=null;
    repaint();
    return true;
    }
   else
    return false;
  }
/******
 * Carica un grafo nel pannello prelevando dal Reader r la matrice di adiacenza il cui formato
 * prevede che la prima riga letta contenga il numero di nodi e la successiva riga contiene l'elenco dei nomi dei nodi
 * e nelle righe successive sono conservati i valori della matrice di adiacenza
 * dove le colonne e le righe conservano lo stesso ordine dei nomi dei nodi
 * Successivamente puo' esserci la matrice dei pesi
 * dove le colonne e le righe conservano lo stesso ordine dei nomi dei nodi
 * e gli elementi sono i valori dei singoli archi separati da '|'
 * In seguito, se c'e' un'altra riga, contiene una terna  di valori separati da virgola per ogni nodo
 *  contenente il  valore del codice del colore in decimale e le coordinate x e y del nodo nel grafo.
 * Infine dopo puo' esserci una matrice contenente i valori dei colori degli archi.
 * Nel secondo formato invece sono elencati su ogni riga la coppia di nodi connessi da un arco ed eventualmente il costo dell'arco.
 * Il nome del primo nodo non può essere un numero intero
 * @param r il Reader da cui viene prelevato il grafo
 *******/
public void load(Reader r)
  {
  load(r,false);
  }
/******
 * Carica un grafo nel pannello prelevando dal Reader r la descrizione di un grafo in due formati. Nel primo il formato
 * prevede che la prima riga letta contenga il numero di nodi e la successiva riga contiene l'elenco dei nomi dei nodi
 * e nelle righe successive sono conservati i valori della matrice di adiacenza
 * dove le colonne e le righe conservano lo stesso ordine dei nomi dei nodi
 * Successivamente puo' esserci la matrice dei pesi
 * dove le colonne e le righe conservano lo stesso ordine dei nomi dei nodi
 * e gli elementi sono i valori dei singoli archi separati da '|'
 * In seguito, se c'e' un'altra riga, contiene una terna  di valori separati da virgola per ogni nodo
 * contenente il  valore del codice del colore in decimale e le coordinate x e y del nodo nel grafo.
 * Infine dopo puo' esserci una matrice contenente i valori dei colori degli archi.
 * Nel secondo formato invece sono elencati su ogni riga la coppia di nodi connessi da un arco ed eventualmente il costo dell'arco.
 * Il nome del primo nodo non può essere un numero intero
 * @param r il Reader da cui viene prelevato il grafo
 * @param merge <b>true</b> se il grafo va sovrapposto a quello esistente, <b>false</b> altrimenti
 *******/
public void load(Reader r, boolean merge)
  {
  BufferedReader br=new BufferedReader(r);
  NodeSign<TN> on[]=null;
  ArcSign<TN,TA> oa[][]=null;
  if (merge)
    {
    g=new Graph<TN,TA>(br,g);
    on=n;
    oa=a;
    }
   else
    g=new Graph<TN,TA>(br);
  setGraph(g);
  String line=null;
  try
    {
    line=br.readLine();
    }
   catch (IOException ioe)
    {
    }
  if (line!=null)
    {
    StringTokenizer st=new StringTokenizer(line);
    for (int i=0;i<n.length && st.hasMoreTokens();i++)
      {
      if (merge)
        {
        for (;i<on.length;i++)
          {
          n[i]=on[i];
          }
        }
      try
        {
        StringTokenizer lineT=new StringTokenizer(st.nextToken(),",");
        Color col=new Color(parseInt(lineT.nextToken()));
        int x=Integer.parseInt(lineT.nextToken());
        int y=Integer.parseInt(lineT.nextToken());
        int dim=defaultSize;
        int shape=defaultShape;
        int thick=defaultThickness;
        boolean fill=defaultFill;
        boolean fix=defaultFixing;
        boolean visib=true;
        Image im=null;
        String imn=null;
        if (lineT.hasMoreTokens())
          {
          dim=parseInt(lineT.nextToken());
          }
        if (lineT.hasMoreTokens())
          {
          String last=lineT.nextToken();
          if (lineT.hasMoreTokens())
            {
            shape=parseInt(last);
            fill=(parseInt(lineT.nextToken()))==0?false:true;
            }
          if (lineT.hasMoreTokens())
            {
            last=lineT.nextToken();
            thick=parseInt(last);
            if (lineT.hasMoreTokens())
              {
              visib=(parseInt(lineT.nextToken()))==0?false:true;
              if (lineT.hasMoreTokens())
                last=lineT.nextToken();
               else
                last=null;
              }
            }
          if (ir!=null && last!=null)
            {
            imn=last;
            im=ir.getImage(imn);
            }
          }
        n[i].setX(x);
        n[i].setY(y);
        n[i].setColor(col);
        n[i].setSize(dim);
        n[i].setShape(shape);
        n[i].setFilled(fill);
        n[i].setThickness(thick);
        n[i].setVisible(visib);
        n[i].setImage(imn,im);
//        System.out.println(n[i]);
        }
       catch (NumberFormatException nfe)
        {
        }
      }
    for (int i=0;i<a.length;i++)
      {
      try
        {
        line=br.readLine();
        }
       catch (IOException ioe)
        {
        break;
        }
      if (line!=null)
        {
        st=new StringTokenizer(line);
        if (merge)
          {
          for (;i<on.length;i++)
            {
            int j;
            for (j=0;j<on.length;j++)
              a[i][j]=oa[i][j];
            for (;j<n.length;j++)
              a[i][j]=null;
            }
          }
        for (int j=0;st.hasMoreTokens() && j<a[i].length;j++)
          {
          if (merge)
            {
            for (;j<on.length;j++)
              {
              a[i][j]=null;
              }
            }
          Arc<TN,TA> ap=g.getArc(i,j);
          ArcSign<TN,TA> last=null;
          a[i][j]=null;
          StringTokenizer vals=new StringTokenizer(st.nextToken(),"|");
          Color col=Color.black;
          boolean vis=true;
          int thick=0;
          for (;ap!=null;ap=ap.next)
            {
            if (vals.hasMoreTokens())
              {
              StringTokenizer arc=new StringTokenizer(vals.nextToken(),",");
              if (arc.hasMoreTokens())
                col=new Color(parseInt(arc.nextToken()));
              if (arc.hasMoreTokens())
                thick=parseInt(arc.nextToken());
              if (arc.hasMoreTokens())
                vis=(parseInt(arc.nextToken())!=0?true:false);
              }
            if (last==null)
              {
              a[i][j]=new ArcSign<TN,TA>(n[i],ap,n[j],col,thick,vis,null);
              a[i][j].addGraphListener(this);
              last=a[i][j];
              }
             else
              {
              last.next=new ArcSign<TN,TA>(n[i],ap,n[j],col,thick,vis,null);
              last.next.addGraphListener(this);
              last=last.next;
              }
            }
          }
        }
       else
        break;
      }
    try
      {
      line=null;
      line=br.readLine();
      }
     catch (IOException ioe)
      {
      }
    if (line!=null)
      {
      if (!merge || bgImage==null)
        setBackImage(line);
      }
     else
      setBackImage((Image)null);
    }
  if (merge)
    setModified(true);
   else
    setModified(false);
  }
/******
 * converte una stringa contenente un valore intero nelle diverse basi
 *
 * @param val la stringa contenente il valore intero
 * @return il valore convertito in intero
 ******/
private int parseInt(String val)
  {
  try
    {
    int ris;
  //  System.out.print("<"+val+">");
    if (val.startsWith("0x"))
      ris=Integer.parseInt(val.substring(2),0x10);
     else if (val.startsWith("0b"))
      ris=Integer.parseInt(val.substring(2),0b10);
     else if (val.startsWith("0"))
      ris=Integer.parseInt(val,010);
     else
      ris=Integer.parseInt(val);
    return ris;
    }
   catch (NumberFormatException nfe)
    {
    return 0;
    }
  }
/******
 * Salva la matrice di adiacenza del grafo nel pannello su un Writer  in un formato che
 * prevede che la prima riga scritta contenga il numero di nodi e la successiva riga contenga l'elenco dei nomi
 * e delle caratteristiche dei nodi.
 * Nelle righe successive sono conservati i valori della matrice di adiacenza
 * dove le colonne e le righe conservano lo stesso ordine dei nomi dei nodi
 * Successivamente c'e' la matrice dei pesi
 * dove le colonne e le righe conservano lo stesso ordine dei nomi dei nodi
 * e gli elementi sono i valori dei singoli archi separati da '|'
 * In seguito c'e' una riga che contiene una terna  di valori separati da virgola per ogni nodo
 * contenente il  valore del codice del colore in decimale e le coordinate x e y del nodo nel grafo.
 * Infine dopo c'e' una matrice contenente i valori dei colori degli archi.
 * @param w il Writer su cui viene salvato il grafo
 *******/
public void save(Writer w)
  {
  g.save(w);
  PrintWriter pw=new PrintWriter(w);
  for (int i=0;i<a.length;i++)
    {
    if (n[i].getImageName()==null)
      pw.format("0x%x,%d,%d,%d,%d,%d,%d,%d ",n[i].getColor().getRGB()&0xffffff,n[i].getX(),n[i].getY(),n[i].getSize(),n[i].getShape(),(n[i].isFilled())?1:0,n[i].getThickness(),(n[i].isVisible())?1:0);
//      pw.format("0x%x,%d,%d,%d ",n[i].getColor().getRGB()&0xffffff,n[i].getX(),n[i].getY(),n[i].getSize());
//      pw.print((n[i].getColor().getRGB()&0xffffff)+","+n[i].getX()+","+n[i].getY()+","+n[i].getSize()+" ");
     else
      pw.format("0x%x,%d,%d,%d,%d,%d,%d,%d,%s ",n[i].getColor().getRGB()&0xffffff,n[i].getX(),n[i].getY(),n[i].getSize(),n[i].getShape(),(n[i].isFilled())?1:0,n[i].getThickness(),(n[i].isVisible())?1:0,n[i].getImageName());
//      pw.format("0x%x,%d,%d,%d,%s ",n[i].getColor().getRGB()&0xffffff,n[i].getX(),n[i].getY(),n[i].getSize(),n[i].getImageName());
//      pw.print((n[i].getColor().getRGB()&0xffffff)+","+n[i].getX()+","+n[i].getY()+","+n[i].getSize()+","+n[i].getImageName()+" ");
    }
  pw.println();
  for (int i=0;i<a.length;i++)
    {
    for (int j=0;j<a[i].length;j++)
      {
      if (a[i][j]!=null)
        {
        pw.format("0x%x,%d,%d",a[i][j].getColor().getRGB()&0xffffff,a[i][j].th,(a[i][j].visible?1:0));
//        pw.print(""+a[i][j].getColor().getRGB()&0xffffff);
        for (ArcSign<TN,TA> asp=a[i][j].next;asp!=null;asp=asp.next)
          pw.format("|0x%x,%d,%d",asp.getColor().getRGB()&0xffffff,a[i][j].th,(a[i][j].visible?1:0));
//        pw.print("|"+(asp.getColor().getRGB()&0xffffff));
        pw.print(" ");
        }
       else
        pw.print("0 ");
      }
    pw.println();
    }
  if (bgName!=null)
    pw.println(bgName);
  pw.flush();
  setModified(false);
  }
/********
 * Cancella il grafo nel GraphPanel
 ********/
public void clear()
  {
  clear(true);
  }
/********
 * Cancella il grafo nel GraphJPanel. Non attiva i GraphListeners
 * @param erase <b>true</b> se deve essere cancellato anche lo sfondo, <b>false</b> altrimenti
 ********/
public void clear(boolean erase)
  {
  if (erase)
    {
    bgImage=null;
    bgName=null;
    }
  g=new Graph<TN,TA>();
  setGraph(g);
  setModified(false);
  }
/******
 * Inizializza il GraphPanel con un grafo
 * @param name nomi dei nodi del grafo
 * @param matAd matrice di adiacenza del grafo
 *******/
public void setGraph(String name[], int matAd[][])
  {
  setGraph(name, matAd, null);
  }
/******
 * Inizializza il GraphPanel con un grafo pesato
 * @param name nomi dei nodi del grafo
 * @param matAd matrice di adiacenza del grafo
 * @param prices matrice dei prezzi di ogni arco
 *******/
public void setGraph(String name[], int matAd[][], double prices[][][])
  {
  g=new Graph<TN,TA>(name,matAd,prices);
  setGraph(g);
  }
/*******
 * Visualizza un grafo sul GraphPanel
 * @param g il grafo da visualizzare
 *******/
public void setGraph(Graph<TN,TA> g)
  {
  setGraph(g,null);
  }
/*******
 * Visualizza un grafo sul GraphPanel
 * @param g il grafo da visualizzare
 * @param c le coordinate dei nodi (primo indice orizzontale, secondo indice verticale)
 *******/
public void setGraph(Graph<TN,TA> g, int c[][])
  {
  setGraph(g,c,Color.black);
  }
/*******
 * Visualizza un grafo sul GraphPanel
 * @param g il grafo da visualizzare
 * @param c le coordinate dei nodi (primo indice orizzontale, secondo indice verticale)
 * @param col il colore dei nodi e degli archi
 *******/
@SuppressWarnings({"rawtypes","unchecked"})
public void setGraph(Graph<TN,TA> g, int c[][], Color col)
  {
  this.g=g;
  n=(NodeSign<TN>[])(new NodeSign[g.degree()]);
  if (c==null || c.length<n.length)
    {
    c=new int[g.degree()][2];
    double deg=0;
    if (n.length!=0)
      deg=2*Math.PI/n.length;
//    size=(getSize().width>getSize().height)?getSize().width:getSize().height;
    int r=size/2-defaultSize;
    for (int i=0;i<n.length;i++)
      {
      int x0=(int)(defaultSize+r+r*Math.cos(i*deg));
      int y0=(int)(defaultSize+r+r*Math.sin(i*deg));
      c[i][0]=x0;
      c[i][1]=y0;
      }
    }
  for (int i=0;i<n.length;i++)
    {
    n[i]=new NodeSign<TN>(g.getNode(i),c[i][0],c[i][1],col,defaultSize);
    n[i].addGraphListener(this);
    }
  a=(ArcSign<TN,TA>[][])(new ArcSign[n.length][n.length]);
  for (int i=0;i<n.length;i++)
    for (int j=0;j<n.length;j++)
      {
      Arc<TN,TA> ap=g.getArc(i,j);
      ArcSign<TN,TA> last=null;
      a[i][j]=null;
      for (;ap!=null;ap=ap.next)
        if (last==null)
          {
          a[i][j]=new ArcSign<TN,TA>(n[i],ap,n[j],col,defaultThickness,true,null);
          a[i][j].addGraphListener(this);
          last=a[i][j];
          }
         else
          {
          last.next=new ArcSign<TN,TA>(n[i],ap,n[j],col,defaultThickness,true,null);
          last.next.addGraphListener(this);
          last=last.next;
          }
      }
  repaint();
//  setModified(true);
  }

/***********
 * Controlla se i due indici di nodi sono validi per il grafo
 * @param i la riga dell'arco
 * @param j la colonna dell'arco
 * @return <b>true</b> se le coordinate dell'arco sono valide, <b>false</b> altrimenti
 ************/
public boolean coordOk(int i, int j)
  {
  return i>=0 && j>=0 && i<n.length && j<n.length;
  }

/***********
 * Ritorna la dimensione preferita del GraphPanel
 * @return la dimensione preferita del GraphPanel
 ************/
public Dimension getPreferredSize()
  {
  int verS=prefSize,horS=prefSize;
  for (int in=0;in<n.length;in++)
    {
    if (n[in].getX()+n[in].getSize()+1>horS)
      horS=n[in].getX()+n[in].getSize()+1;
    if (n[in].getY()+n[in].getSize()+1>verS)
      verS=n[in].getY()+n[in].getSize()+1;
    }
  return new Dimension(horS,verS);
//  return new Dimension(prefSize+defaultSize,prefSize+defaultSize);
  }

/***********
 * Modifica la dimensione del GraphPanel
 ************/
public void setSize(int w, int h)
  {
  super.setSize(w,h);
//  System.out.println("resizing1 "+getSize().width+" "+getSize().height);
  size=(getSize().width>getSize().height)?getSize().width:getSize().height;
  repaint();
  notifySetting();
  }
 
/***********
 * Modifica la dimensione del GraphPanel
 ************/
public void setSize(Dimension d)
  {
  super.setSize(d);
//  System.out.println("resizing2 "+getSize().width+" "+getSize().height);
  size=(getSize().width>getSize().height)?getSize().width:getSize().height;
  repaint();
  notifySetting();
  }

/********
 * Ritorna l'indice del nodo dato il nome
 * @param name nome del nodo
 * @return l'indice del nodo nel grafo, -1 se non esiste nodo con il nome dato
 ********/
public int findNode(String name)
  {
  return g.findNode(name);
  }

/********
 * Risistema i nodi del grafo visualizzato
 ********/
synchronized void relax()
  {
  for (int i = 0 ; i < a.length ; i++)
    for (int j=0; j<a[i].length;j++)
      {
      ArcSign<TN,TA> e = a[i][j];
      if (e==null)
        continue;
    //    System.out.println("relax "+e);
    //    double vx = nodes[e.to].x - nodes[e.from].x;
      double vx = e.to.getX() - e.from.getX();
      double vy = e.to.getY() - e.from.getY();
      double len = Math.sqrt(vx * vx + vy * vy);
      len = (len == 0) ? .0001 : len;
      double f = (a[i][j].getLength() - len) / (len * 3);
      double dx = f * vx;
      double dy = f * vy;
      e.to.addDx(dx);
      e.to.addDy(dy);
      e.from.addDx(-dx);
      e.from.addDy(-dy);
      }
  for (int i = 0 ; i < n.length ; i++)
    {
    NodeSign<TN> n1 = n[i];
//    System.out.println("relax nnode "+n1);
    double dx = 0;
    double dy = 0;
    for (int j = 0 ; j < n.length ; j++)
      {
      if (i == j)
        {
        continue;
        }
      NodeSign<TN> n2 = n[j];
      double vx = n1.getX() - n2.getX();
      double vy = n1.getY() - n2.getY();
      double len = vx * vx + vy * vy;
      if (len == 0)
        {
        dx += Math.random();
        dy += Math.random();
        }
       else if (len < 100*100)
        {
        dx += vx / len;
        dy += vy / len;
        }
      }
    double dlen = dx * dx + dy * dy;
    if (dlen > 0)
      {
      dlen = Math.sqrt(dlen) / 2;
      n1.addDx(dx / dlen);
      n1.addDy(dy / dlen);
      }
    }
  Dimension d = getSize();
  for (int i = 0 ; i < n.length ; i++)
    {
    NodeSign<TN> n0 = n[i];
    if (!n0.isFixed())
      {
      n0.setX((int)(n0.getX() + Math.max(-5, Math.min(5, n0.getDx()))));
      n0.setY((int)(n0.getY() + Math.max(-5, Math.min(5, n0.getDy()))));
      }
    /*
    if (n0.getX() < 0)
      {
      n0.setX(0);
      }
     else if (n0.getX() > d.width)
      {
      n0.setX(d.width);
      }
    if (n0.getY() < 0)
      {
      n0.setY(0);
      }
     else if (n0.getY() > d.height)
      {
      n0.setY(d.height);
      }
    */
    n0.setDx(n0.getDx()/2);
    n0.setDy(n0.getDy()/2);
    }
  repaint();
  }

/********************
 * Attiva o disattiva il processo di adattamento della posizione dei nodi per consentire una corretta collocazione e
 * dare una lunghezza richiesta agli archi
 * @param ad <b>true</b> se il pannello deve adattarsi, <b>false</b> altrimenti
 ********************/
public void setAdaptive(boolean ad)
  {
  if (adaptive==ad)
    return;
   else
    {
    adaptive=ad;
    if (adaptive)
      {
      relaxer=new Thread()
            {
            public void run()
              {
              Thread me = Thread.currentThread();
              while (relaxer == me && isDisplayable() && adaptive)
                {
                relax();
                try
                  {
                  Thread.sleep(100);
                  }
                 catch (InterruptedException e)
                  {
                  break;
                  }
                }
              }
            };
      relaxer.start();
      }
    }
  }

/********************
 * Dice se e' disattivato il processo di adattamento della posizione dei nodi per consentire una corretta collocazione e
 * dare una lunghezza richiesta agli archi
 * @return <b>true</b> se il pannello deve adattarsi, <b>false</b> altrimenti
 ********************/
public boolean isAdaptive()
  {
  return adaptive;
  }

/********
 * Disegna un cammino nel colore indicato
 * Aggiunge gli archi se non presenti
 * @param c il colore del percorso
 * @param steps l'elenco dei nodi che costituiscono il percorso
 ********/
public void drawPath(Color c,String steps[])
  {
  drawPath(c,steps,true);
  }

/********
 * Disegna un cammino nel colore indicato
 * Aggiunge gli archi se non presenti
 * @param c il colore del percorso
 * @param steps l'elenco dei nodi che costituiscono il percorso
 * @param visible <b>true</b> se il percorso deve essere visibile, <b>false</b> altrimenti
 ********/
public void drawPath(Color c,String steps[],boolean visible)
  {
  drawPath(c,c,steps,visible);
  }

/********
 * Disegna un cammino nel colore indicato
 * Aggiunge gli archi se non presenti
 * @param c il colore del percorso
 * @param cn il colore dei nodi (se null il colore non viene cambiato)
 * @param steps l'elenco dei nodi che costituiscono il percorso
 * @param visible <b>true</b> se il percorso deve essere visibile, <b>false</b> altrimenti
 ********/
public void drawPath(Color c,Color cn,String steps[],boolean visible)
  {
  if (steps==null) return;
  int stp[]=new int[steps.length];
  for (int i=0;i<steps.length;i++)
    {
    stp[i]=findNode(steps[i]);
    }
  drawPath(c,cn,stp,visible);
  }

/********
 * Disegna un cammino nel colore indicato
 * Aggiunge gli archi se non presenti
 * @param c il colore del percorso
 * @param steps l'elenco degli indici dei nodi che costituiscono il percorso
 ********/
public void drawPath(Color c,int steps[])
  {
  drawPath(c,steps,true);
  }

/********
 * Disegna un cammino nel colore indicato
 * Aggiunge gli archi se non presenti
 * @param c il colore del percorso
 * @param steps l'elenco degli indici dei nodi che costituiscono il percorso
 * @param visible <b>true</b> se il percorso deve essere visibile, <b>false</b> altrimenti
 ********/
public void drawPath(Color c,int steps[],boolean visible)
  {
  drawPath(c,c,steps,visible);
  }

/********
 * Disegna un cammino nei colori indicati.
 * Aggiunge gli archi se non presenti
 * @param c il colore del percorso
 * @param cn il colore dei nodi (se null il colore non viene cambiato)
 * @param steps l'elenco degli indici dei nodi che costituiscono il percorso
 * @param visible <b>true</b> se il percorso deve essere visibile, <b>false</b> altrimenti
 ********/
public void drawPath(Color c,Color cn,int steps[],boolean visible)
  {
  if (steps==null) return;
  for (int i=0;i<steps.length-1;i++)
    {
    int im=steps[i];
    int it=steps[i+1];
    if (coordOk(im,it))
      {
      n[im].setVisible(visible);
      if (cn!=null)
        {
        n[im].setColor(cn);
        }
      if (a[im][it]!=null)
        for (ArcSign<TN,TA> asp=a[im][it];asp!=null;asp=asp.next)
          {
          asp.setColor(c);
          asp.setVisible(visible);
          }
       else
        if (visible)
          {
          a[im][it]=new ArcSign<TN,TA>(n[im],g.addArc(n[im].nn.name,n[it].nn.name),n[it],c,defaultThickness,visible,null);
          a[im][it].addGraphListener(this);
          }
      setModified(true);
      }
     else
      break;
    }
  n[steps[steps.length-1]].setVisible(visible);
  n[steps[steps.length-1]].setColor(cn);
  repaint();
  }

/********
 * Seleziona o deselaziona tutti i nodi del grafo
 * @param sel <b>true</b> se seleziona, <b>false</b> se deseleziona
 ********/
 public void selectAll(boolean sel)
   {
   for (NodeSign<TN> nn:n)
     nn.selected=sel;
   repaint();
   }
  
/********
 * Modifica la dimensione di un nodo nel grafo
 * @param name il nome del nodo
 * @param newSize la nuova dimensione del nodo
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ********/
public boolean resizeNode(String name,int newSize)
  {
  int res;
  defaultSize=newSize;
  notifySetting();
  if ((res=g.findNode(name))>=0)
    {
    if (n[res].getSize()!=newSize)
      {
      n[res].setSize(newSize);
      setModified(true);
      repaint();
      return true;
      }
    }
  return false;
  }

/********
 * Modifica lo spessore di un nodo nel grafo
 * @param name il nome del nodo
 * @param newThickness la nuova dimensione del nodo
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ********/
public boolean setThickness(String name,int newThickness)
  {
  int res;
  defaultThickness=newThickness;
  notifySetting();
  if ((res=g.findNode(name))>=0)
    {
    if (n[res].getThickness()!=newThickness)
      {
      n[res].setThickness(newThickness);
      setModified(true);
      repaint();
      return true;
      }
    }
  return false;
  }

/********
 * Modifica la forma di un nodo nel grafo
 * @param name il nome del nodo
 * @param newShape la nuova forma del nodo
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ********/
public boolean reshapeNode(String name,int newShape)
  {
  int res;
  defaultShape=newShape;
  notifySetting();
  if ((res=g.findNode(name))>=0)
    {
    if (n[res].getShape()!=newShape)
      {
      n[res].setShape(newShape);
      setModified(true);
      repaint();
      return true;
      }
    }
  return false;
  }

/********
 * Modifica il riempimento di un nodo nel grafo
 * @param name il nome del nodo
 * @param newFill il nuovo stato di riempimento del nodo
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ********/
public boolean changeNodeFill(String name,boolean newFill)
  {
  int res;
  defaultFill=newFill;
  notifySetting();
  if ((res=g.findNode(name))>=0)
    {
    if (n[res].isFilled()!=newFill)
      {
      n[res].setFilled(newFill);
      setModified(true);
      repaint();
      return true;
      }
    }
  return false;
  }

/********
 * Modifica il fissaggio di un nodo nel grafo
 * @param name il nome del nodo
 * @param newFix il nuovo stato di fissaggio del nodo
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ********/
public boolean changeNodeFix(String name,boolean newFix)
  {
  int res;
  defaultFixing=newFix;
  notifySetting();
  if ((res=g.findNode(name))>=0)
    {
    if (n[res].isFixed()!=newFix)
      {
      n[res].setFixed(newFix);
      setModified(true);
      repaint();
      return true;
      }
    }
  return false;
  }

/********
 * Modifica lo spessore del bordo di un nodo nel grafo
 * @param name il nome del nodo
 * @param newThick il nuovo spessore del nodo
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ********/
public boolean changeNodeThickness(String name,int newThick)
  {
  int res;
  defaultThickness=newThick;
  notifySetting();
  if ((res=g.findNode(name))>=0)
    {
    if (newThick!=n[res].getThickness())
      {
      n[res].setThickness(newThick);
      setModified(true);
      repaint();
      return true;
      }
    }
  return false;
  }

/********
 * Modifica il fissaggio di un nodo nel grafo
 * @param name il nome del nodo
 * @param newFix il nuovo fissaggio del nodo
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ********/
public boolean changeNodeFixing(String name,boolean newFix)
  {
  int res;
  defaultFixing=newFix;
  notifySetting();
  if ((res=g.findNode(name))>=0)
    {
    if (n[res].isFixed()!=newFix)
      {
      n[res].setFixed(newFix);
      setModified(true);
      repaint();
      return true;
      }
    }
  return false;
  }

/********
 * Modifica l'immagine di un nodo nel grafo
 * @param name il nome del nodo
 * @param image la nuova immagine del nodo
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ********/
public boolean changeNodeImage(String name,String image)
  {
  boolean res=false;
  int ind=-1;
  defaultImage=image;
  notifySetting();
  if ((ind=findNode(name))>=0)
    {
    if (image==null && n[ind].getImageName()!=null)
      {
      n[ind].setImage(null,null);
      setModified(true);
      repaint();
      res=true;
      }
     else if(image!=null && !image.equals(n[ind].getImageName()))
      {
      n[ind].setImage(image,getImage(image));
      setModified(true);
      repaint();
      res=true;
      }
    }
  return res;
  }

/********
 * Rinomina un nodo nel grafo
 * @param name il vecchio nome del nodo
 * @param newName il nuovo nome del nodo
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ********/
public boolean renameNode(String name,String newName)
  {
  boolean res;
  if (res=g.renameNode(name,newName))
    {
    setModified(true);
    repaint();
    }
  return res;
  }

/*****************
 * Rimuove il nodo selezionato dal grafo 
 * @param x l'ascissa del punto di selezione
 * @param y l'ordinata del punto di selezione
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ******************/
public boolean removeNode(int x,int y)
  {
  String n=getNodeName(x,y);
  if (n!=null)
    {
    removeNode(n);
    return true;
    }
   else
    return false;
  }

/********
 * Rimuove un nodo col dato nome dal grafo
 * @param name nome del nodo da eliminare
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ********/
@SuppressWarnings({"rawtypes","unchecked"})
public boolean removeNode(String name)
  {
//  System.out.println("removing "+name);
  int pos=-1;
  pos=findNode(name);
  if (g.removeNode(name))
    {
    NodeSign<TN> ns=getNode(name);
    ArcSign<TN,TA> arcs[]=a[pos];
    NodeSign<TN> nn[]=(NodeSign<TN>[])(new NodeSign[n.length-1]);
    ArcSign<TN,TA> aa[][]=(ArcSign<TN,TA>[][])(new ArcSign[n.length-1][n.length-1]);
    for (int i=0,k=0;i<n.length;i++)
      {
      if (i==pos)
        {
        continue;
        }
      if (k==nn.length) break;
      nn[k]=n[i];
      int j,h;
      for (j=0,h=0;j<n.length;j++)
        {
        if (j==pos)
          continue;
        if (h==nn.length) break;
        aa[k][h]=a[i][j];
        h++;
        }
      k++;
      }
    n=nn;
    a=aa;
    setModified(true);
    if (ns!=null)
      {
//      System.out.println("Segnalo la rimozione del nodo "+ns);
      for (int i=0;ns!=null && i<graphListeners.size();i++)  // per attivare i listeners solo alla selezione interattiva
        graphListeners.elementAt(i).nodeRemoved(new GraphEvent(this,ns));
      for (int j=0,h=0;j<arcs.length;j++)
        {
        ArcSign<TN,TA> as=arcs[j];
//        System.out.println("Segnalo la rimozione dell'arco "+j+" "+as);
        for (int i=0;as!=null && i<graphListeners.size();i++)  // per attivare i listeners solo alla selezione interattiva
          graphListeners.elementAt(i).arcRemoved(new GraphEvent(this,as));
        }
      }
    repaint();
//    System.out.println("removed "+name);
    return true;
    }
   else
    return false;
  }

/*********
 * Aggiunge un nodo col dato nome al grafo
 * @param name nome del nodo da aggiungere
 * @return true se il grafo e' cambiato, false altrimenti
 *********/
public boolean addNode(String name)
  {
  return addNode(name,defaultColor);
  }

/*********
 * Aggiunge un nodo col dato nome al grafo di un certo colore
 * @param name nome del nodo da aggiungere
 * @param c colore del nodo
 * @return true se il grafo e' cambiato, false altrimenti
 *********/
public boolean addNode(String name,Color c)
  {
  int xn=size/2,yn=size/2;
  for (int i=0;i<n.length;i++)
    {
    if (Math.abs(n[i].x-xn)<defaultSize && Math.abs(n[i].y-yn)<defaultSize)
      {
      if (xn!=n[i].x)
        xn+=(xn-n[i].x)*2*defaultSize/Math.abs(xn-n[i].x);
       else
        xn+=2*defaultSize;
      if (yn!=n[i].y)
        yn-=(yn-n[i].y)*2*defaultSize/Math.abs(yn-n[i].y);
       else
        yn-=2*defaultSize;
      if (xn<0) xn=0;
      if (yn<0) yn=0;
      if (xn>=size) xn=size;
      if (yn>=size) yn=size;
      }
    }
  return addNode(name,xn,yn,c);
  }

/*********
 * Aggiunge un nodo col dato nome al grafo in una data posizione
 * @param name nome del nodo da aggiungere
 * @param x la coordinata x del nodo
 * @param y la coordinata y del nodo
 * @return true se il grafo e' cambiato, false altrimenti
 *********/
public boolean addNode(String name, int x, int y)
  {
  return addNode(name,x,y,Color.black);
  }

/*********
 * Aggiunge un nodo col dato nome al grafo di un certo colore in una data posizione
 * @param name nome del nodo da aggiungere
 * @param x la coordinata x del nodo
 * @param y la coordinata y del nodo
 * @param c colore del nodo
 * @return true se il grafo e' cambiato, false altrimenti
 *********/
public boolean addNode(String name, int x, int y, Color c)
  {
  return addNode(name,x,y,c,defaultSize);
  }

/*********
 * Aggiunge un nodo col dato nome al grafo di un certo colore in una data posizione
 * @param name nome del nodo da aggiungere
 * @param x la coordinata x del nodo
 * @param y la coordinata y del nodo
 * @param c colore del nodo
 * @param dim dimensione del nodo
 * @return true se il grafo e' cambiato, false altrimenti
 *********/
public boolean addNode(String name, int x, int y, Color c,int dim)
  {
  return addNode(name,x,y,c,dim,null);
  }

/*********
 * Aggiunge un nodo col dato nome al grafo di un certo colore in una data posizione
 * @param name nome del nodo da aggiungere
 * @param x la coordinata x del nodo
 * @param y la coordinata y del nodo
 * @param c colore del nodo
 * @param dim dimensione del nodo
 * @param image immagine di sfondo del nodo
 * @return true se il grafo e' cambiato, false altrimenti
 *********/
public boolean addNode(String name,int x,int y,Color c,int dim,String image)
  {
  return addNode(name,x,y,c,dim,image,defaultShape,defaultFill,defaultThickness);
  }
/*********
 * Aggiunge un nodo col dato nome al grafo di un certo colore in una data posizione
 * @param name nome del nodo da aggiungere
 * @param x la coordinata x del nodo
 * @param y la coordinata y del nodo
 * @param c colore del nodo
 * @param dim dimensione del nodo
 * @param shape forma del nodo
 * @return true se il grafo e' cambiato, false altrimenti
 *********/
public boolean addNode(String name,int x,int y,Color c,int dim,int shape)
  {
  return addNode(name,x,y,c,dim,shape,defaultFill);
  }
/*********
 * Aggiunge un nodo col dato nome al grafo di un certo colore in una data posizione
 * @param name nome del nodo da aggiungere
 * @param x la coordinata x del nodo
 * @param y la coordinata y del nodo
 * @param c colore del nodo
 * @param dim dimensione del nodo
 * @param shape forma del nodo
 * @param filled <b>true</b> se il nodo è pieno, <b>false</b> se il nodo è vuoto
 * @return true se il grafo e' cambiato, false altrimenti
 *********/
public boolean addNode(String name,int x,int y,Color c,int dim,int shape,boolean filled)
  {
  return addNode(name,x,y,c,dim,shape,filled,defaultThickness);
  }
/*********
 * Aggiunge un nodo col dato nome al grafo di un certo colore in una data posizione
 * @param name nome del nodo da aggiungere
 * @param x la coordinata x del nodo
 * @param y la coordinata y del nodo
 * @param c colore del nodo
 * @param dim dimensione del nodo
 * @param shape forma del nodo
 * @param filled <b>true</b> se il nodo è pieno, <b>false</b> se il nodo è vuoto
 * @param thickness lo spessore del bordo del segno 
 * @return true se il grafo e' cambiato, false altrimenti
 *********/
public boolean addNode(String name,int x,int y,Color c,int dim,int shape,boolean filled,int thickness)
  {
  return addNode(name,x,y,c,dim,null,shape,filled,thickness);
  }
/*********
 * Aggiunge un nodo col dato nome al grafo di un certo colore in una data posizione
 * @param name nome del nodo da aggiungere
 * @param x la coordinata x del nodo
 * @param y la coordinata y del nodo
 * @param c colore del nodo
 * @param dim dimensione del nodo
 * @param image immagine di sfondo del nodo, <b>null</b> se il nodo è una forma geometrica
 * @param shape forma del nodo
 * @param filled <b>true</b> se il nodo è pieno, <b>false</b> se il nodo è vuoto
 * @param thickness lo spessore del bordo del segno 
 * @return true se il grafo e' cambiato, false altrimenti
 *********/
public boolean addNode(String name,int x,int y,Color c,int dim,String image,int shape,boolean filled, int thickness)
  {
  return addNode(name,x,y,c,dim,image,shape,filled,thickness,defaultFixing);
  }
/*********
 * Aggiunge un nodo col dato nome al grafo di un certo colore in una data posizione
 * @param name nome del nodo da aggiungere
 * @param x la coordinata x del nodo
 * @param y la coordinata y del nodo
 * @param c colore del nodo
 * @param dim dimensione del nodo
 * @param image immagine di sfondo del nodo, <b>null</b> se il nodo è una forma geometrica
 * @param shape forma del nodo
 * @param filled <b>true</b> se il nodo è pieno, <b>false</b> se il nodo è vuoto
 * @param thickness lo spessore del bordo del segno 
 * @param fixing indica se il nodo è fisso o può fluttuare 
 * @return true se il grafo e' cambiato, false altrimenti
 *********/
@SuppressWarnings({"rawtypes","unchecked"})
public boolean addNode(String name,int x,int y,Color c,int dim,String image,int shape,boolean filled, int thickness, boolean fixing)
  {
  if (name!=null)
    name=name.replace(' ','_');
   else
    return false;
  if (!name.equals("") && findNode(name)==-1 && g.addNode(name))
    {
    NodeSign<TN> nn[]=(NodeSign<TN>[])(new NodeSign[n.length+1]);
    ArcSign<TN,TA> aa[][]=(ArcSign<TN,TA>[][])(new ArcSign[n.length+1][n.length+1]);
    for (int i=0;i<n.length;i++)
      {
      nn[i]=n[i];
      for (int j=0;j<n.length;j++)
        {
        aa[i][j]=a[i][j];
        }
      aa[i][n.length]=null;
      aa[n.length][i]=null;
      }
    aa[n.length][n.length]=null;
    if (ir!=null && image!=null)
      {
      nn[n.length]=new NodeSign<TN>(g.getNode(name),x,y,c,dim,ir.getImage(image),image);
      }
     else
      {
      nn[n.length]=new NodeSign<TN>(g.getNode(name),x,y,c,dim,shape,filled,thickness);
      }
    nn[n.length].setFixed(fixing);
    nn[n.length].addGraphListener(this);
    n=nn;
    a=aa;
    setModified(true);
    for (int i=0;i<graphListeners.size();i++)  // per attivare i listeners solo alla selezione interattiva
      graphListeners.elementAt(i).nodeAdded(new GraphEvent(this,nn[n.length-1]));
    repaint();
    return true;
    }
   else
    return false;
  }

/*********
 * Sostituisce un nodo col dato nome con il nodo indicato
 * @param name nome del nodo di cui sostituire la rappresentazione
 * @param ns la nuova rappresentazione del nodo
 * @return true se il grafo e' cambiato, false altrimenti
 *********/
public boolean changeNode(String name,NodeSign<TN> ns)
  {
  name=name.replace(' ','_');
  int nn=findNode(name);
  if (nn!=-1)
    {
    n[nn]=ns;
    setModified(true);
    repaint();
    return true;
    }
   else
    return false;
  }

/*****************
 * Rimuove un arco dal grafo 
 * @param as l'arco da rimuovere
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ******************/
public boolean removeArc(ArcSign<TN,TA> as)
  {
  if (as!=null)
    {
    int i,j;
    ArcSign<TN,TA> prec=null,p=a[i=findNode(as.from.getName())][j=findNode(as.to.getName())];
    for (;p!=null && p!=as;p=p.next)
      prec=p;
    if (prec==null)
      {
      a[i][j]=a[i][j].next;
      }
     else
      {
      prec.next=prec.next.next;
      }
    as.next=null;
    setModified(true);
    if (as!=null)
      for (int k=0;k<graphListeners.size();k++)  // per attivare i listeners solo alla selezione interattiva
        graphListeners.elementAt(k).arcRemoved(new GraphEvent(this,as));
    repaint();
    return true;
    }
   else
    return false;
  }

/*****************
 * Rimuove un arco dal grafo tra il nodo source e il nodo dest
 * @param source la sorgente dell'arco
 * @param dest la destinazione dell'arco
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ******************/
public boolean removeArc(String source, String dest)
  {
  int i=findNode(source),j=findNode(dest);
  if (i!=-1 && j!=-1 && g.removeArc(source,dest))
    {
    ArcSign<TN,TA> as=a[i][j];
    a[i][j]=a[i][j].next;
    as.next=null;
    setModified(true);
    if (as!=null)
      for (int k=0;k<graphListeners.size();k++)  // per attivare i listeners solo alla selezione interattiva
        graphListeners.elementAt(k).arcRemoved(new GraphEvent(this,as));
    repaint();
    return true;
    }
   else
    return false;
  }

/*****************
 * Rimuove gli archi dal grafo tra il nodo source e il nodo dest
 * @param source la sorgente dell'arco
 * @param dest la destinazione dell'arco
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ******************/
public boolean removeArcs(String source, String dest)
  {
  int i=findNode(source),j=findNode(dest);
  if (i!=-1 && j!=-1 && g.removeArc(source,dest))
    {
    ArcSign<TN,TA> as=a[i][j];
    a[i][j]=null;
    setModified(true);
    if (as!=null)
      for (int k=0;k<graphListeners.size();k++)  // per attivare i listeners solo alla selezione interattiva
        graphListeners.elementAt(k).arcRemoved(new GraphEvent(this,as));
    repaint();
    return true;
    }
   else
    return false;
  }

/*****************
 * Rimuove l'arco selezionato dal grafo 
 * @param x l'ascissa del punto di selezione
 * @param y l'ordinata del punto di selezione
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ******************/
public boolean removeArc(int x,int y)
  {
  ArcSign<TN,TA> as=getArc(x,y);
  if (as!=null)
    {
    int i=g.findNode(as.a.f),j=g.findNode(as.a.t);
    if (!g.removeArc(as.a))  // tutti gli archi rimossi
      a[i][j]=null;
     else
      {
      ArcSign<TN,TA> last=null;
      for (ArcSign<TN,TA> asp=a[i][j];asp!=null;last=asp,asp=asp.next)
        if (asp.a==as.a)
          if (last==null)
            a[i][j]=asp.next;
           else
            last.next=asp.next;
      }
    setModified(true);
    for (int k=0;k<graphListeners.size();k++)  // per attivare i listeners solo alla selezione interattiva
      graphListeners.elementAt(k).arcRemoved(new GraphEvent(this,as));
    repaint();
    return true;
    }
   else
    return false;
  }

/*****************
 * Aggiunge un arco al grafo tra il nodo sorg e il nodo dest
 * @param source la sorgente dell'arco
 * @param dest la destinazione dell'arco
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ******************/
public boolean addArc(String source, String dest)
  {
  return addArc(source,dest,defaultColor,defaultWeight);
  }

/*****************
 * Aggiunge un arco al grafo tra il nodo sorg e il nodo dest del colore indicato
 * @param source la sorgente dell'arco
 * @param dest la destinazione dell'arco
 * @param c il colore dell'arco
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ******************/
public boolean addArc(String source, String dest, Color c)
  {
  return addArc(source,dest,c,defaultWeight);
  }

/*****************
 * Aggiunge un arco al grafo tra il nodo sorg e il nodo dest con il costo indicato
 * @param source la sorgente dell'arco
 * @param dest la destinazione dell'arco
 * @param price il costo dell'arco
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ******************/
public boolean addArc(String source, String dest, double price)
  {
  return addArc(source,dest,defaultColor,price);
  }

/*****************
 * Aggiunge un arco al grafo tra il nodo sorg e il nodo dest con il costo indicato del colore indicato
 * @param source la sorgente dell'arco
 * @param dest la destinazione dell'arco
 * @param c il colore dell'arco
 * @param price il costo dell'arco
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ******************/
public boolean addArc(String source, String dest, Color c, double price)
  {
  return addArc(source,dest,c,defaultThickness,true,price);
  }
/*****************
 * Aggiunge un arco al grafo tra il nodo sorg e il nodo dest con il costo indicato del colore indicato
 * @param source la sorgente dell'arco
 * @param dest la destinazione dell'arco
 * @param c il colore dell'arco
 * @param thickness lo spessore dell'arco
 * @param visible la visibilità dell'arco
 * @param price il costo dell'arco
 * @return <b>true</b> se il grafo e' cambiato, <b>false</b> altrimenti
 ******************/
public boolean addArc(String source, String dest, Color c, int thickness, boolean visible, double price)
  {
  addNode(source);
  addNode(dest);
  int da=findNode(source),ad=findNode(dest);
  Arc<TN,TA> newArc=g.addArc(source,dest,price);
  if (newArc!=null)
    {
    a[da][ad]=new ArcSign<TN,TA>(n[da],newArc,n[ad],c,thickness,visible,a[da][ad]);
    ArcSign<TN,TA> as=a[da][ad];
    as.addGraphListener(this);
    setModified(true);
    for (int i=0;i<graphListeners.size();i++)  // per attivare i listeners solo alla selezione interattiva
      graphListeners.elementAt(i).nodeAdded(new GraphEvent(this,as));
    repaint();
    return true;
    }
   else
    return false;
  }

/*****************
 * Modifica il peso di un arco del grafo tra il nodo sorg e il nodo dest con il costo indicato
 * @param source la sorgente dell'arco
 * @param dest la destinazione dell'arco
 * @param price il costo dell'arco
 * @return true se il grafo e' cambiato, false altrimenti
 ******************/
public boolean changeArcWeight(String source, String dest, double price)
  {
  ArcSign<TN,TA> as=getArc(source,dest);
  if (as!=null && as.a.p!=price)
    {
    as.a.p=price;
    setModified(true);
    defaultWeight=price;
    notifySetting();
    repaint();
    return true;
    }
   else
    return false;
  }

/*****************
 * Modifica lo spessore di un arco del grafo tra il nodo sorg e il nodo dest con lo spessore indicato
 * @param source la sorgente dell'arco
 * @param dest la destinazione dell'arco
 * @param thickness il costo dell'arco
 * @return true se il grafo e' cambiato, false altrimenti
 ******************/
public boolean changeArcThickness(String source, String dest, int thickness)
  {
  ArcSign<TN,TA> as=getArc(source,dest);
  if (as!=null && as.th!=thickness)
    {
    as.th=thickness;
    setModified(true);
    defaultThickness=thickness;
    notifySetting();
    repaint();
    return true;
    }
   else
    return false;
  }

int dx=0,dy=0;
int oldx=0,oldy=0;
int posx=0,posy=0;

/*****************
 * Rimuove un nodo in seguito alla scelta da popup
 ******************/
void remNodePU()
  {
  String nn=getNodeName(lastEvent.getX(),lastEvent.getY());
  if (nn!=null)
    {
    removeNode(nn);
    setModified(true);
    }
  for (int nr=0;nr<n.length;nr++)
    {
    if (n[nr].selected)
      {
      if (removeNode(n[nr].getName()))
        {
        nr--;
        setModified(true);
        }
      }
    }
  }

/*****************
 * Prepara la selezione da popup
 ******************/
private void setPU(int op)
  {
  operation=op;
  doOperation(lastEvent);
  }

/*****************
 * Cancella un arco in seguito alla scelta da popup
 ******************/
void unlinkPU()
  {
  if (removeArc(lastEvent.getX(),lastEvent.getY()))
    setModified(true);
  }

/*****************
 * Permette di selezionare un colore tramite un ColorPickPanel
 ******************/
private Color inputColor()
  {
  Component par=this.getParent();
  for (;!(par instanceof Frame);)
    {
//    System.out.println(par);
    par=par.getParent();
    }
  final JDialog d=new JDialog((Frame)par,prompt[PCOLOR],true);
  d.addWindowListener(new WindowAdapter()
    {
    public void windowClosing(WindowEvent we)
      {
      d.setVisible(false);
      d.dispose();
      }
    });
  ColorPickPanel cpp=new ColorPickPanel(defaultColor);
  d.add(cpp);
  d.setBounds(getBounds().x+getWidth()/2-100,getBounds().y+getHeight()/2,100,30);
  d.pack();
  d.setVisible(true);
  return cpp.getColor();
  }

/*****************
 * Permette l'input di una stringa tramite un dialog
 * @param t titolo del dialog
 ******************/
private String inputText(String t)
  {
  return inputText(t,"");
  }
/*****************
 * Permette l'input di una stringa tramite un dialog
 * @param t titolo del dialog
 * @param v valore iniziale del dialog
 * @return la stringa immessa, <b>null</b> se la stringa e' vuota
 ******************/
private String inputText(String t,String v)
  {
  return inputText(t,v,true);
  }
/*****************
 * Permette l'input di una stringa tramite un dialog
 * @param t titolo del dialog
 * @param v valore iniziale del dialog
 * @param selected <b>true</b> se il valore iniziale deve essere selezionato, <b>false</b> altrimenti
 * @return la stringa immessa, <b>null</b> se la stringa e' vuota
 ******************/
private String inputText(String t,String v,boolean selected)
  {
  Component par=this;
  for (;!(par instanceof Frame);)
    {
//    System.out.println(par);
    par=par.getParent();
    }
  final JDialog d=new JDialog((Frame)par,t,true);
  JTextField text=new JTextField(t.length()+5);
  text.setText(v);
  if (selected)
    {
    text.setSelectionStart(0);
    text.setSelectionEnd(text.getText().length());
    }
  text.addActionListener(new ActionListener()
    {
    public void actionPerformed(ActionEvent ae)
      {
      d.setVisible(false);
      d.dispose();
      }
    });
  d.addWindowListener(new WindowAdapter()
    {
    public void windowClosing(WindowEvent we)
      {
      d.setVisible(false);
      d.dispose();
      }
    });
  d.add(text);
  d.setBounds(getBounds().x+getWidth()/2-100,getBounds().y+getHeight()/2,100,30);
  d.pack();
  d.setVisible(true);
//  System.out.println("Passato dopo inputtext");
  String res=text.getText().trim();
  if ("".equals(res))
    return null;
   else
    return res;
  }

/*****************
 * Gestisce le azioni relative ai menu
 * @param ae l'evento da gestire
 ******************/
public void actionPerformed(ActionEvent ae)
  {
  if (ae.getSource()==mnMinLPU)
    setPU(PMINL);
   else if (ae.getSource()==mnMinPPU)
    setPU(PMINP);
   else if (ae.getSource()==mnRenPU)
    setPU(PREN);
   else if (ae.getSource()==mnUnlinkPU)
    unlinkPU();
   else if (ae.getSource()==mnColSetPU)
    setPU(PCOLOR);
   else if (ae.getSource()==mnRemPU)
    remNodePU();
   else if (ae.getSource()==mnWeightPU)
    setPU(PWEIGHT);
   else if (ae.getSource()==mnZoomPU)
    setPU(PZOOM);
   else if (ae.getSource()==mnSizePU)
    setPU(PSIZE);
   else if (ae.getSource()==mnShapePU)
    setPU(PSHAPE);
   else if (ae.getSource()==mnFillPU)
    setPU(PFILL);
   else if (ae.getSource()==mnThickPU)
    setPU(PTHICK);
   else if (ae.getSource()==mnImagePU)
    setPU(PIMAGE);
   else if (ae.getSource()==mnFixPU)
    setPU(PFIX);
   else if (ae.getSource()==mnInfoPU)
    setPU(PINFO);
  setColor(defaultColor);
  setWeight(defaultWeight);
  lastEvent=null;
  repaint();
  }

/*************
 * Gestisce l'evento mouse click se l'evento non e' stato gia' consumato
 * Se il click col tasto destro è su un nodo o un arco presente
 *    se sono attivi i popup viene visualizzato il popup relativo a quel nodo/arco
 *    se ci sono listeners vengono avvertiti della selezione 
 * @param e l'evento del mouse
 *************/
public void mouseClicked(MouseEvent e)
  {
//  System.out.println("mouseclicked dx="+dx+" dy="+dy+" oldx="+oldx+" oldy="+oldy);
  if (e.isConsumed())
    return;
//  System.out.println("Click in GraphPanel");
//  System.out.printf("click b3 %o %o %o %o\n",e.getModifiersEx(),(e.getModifiersEx() & 
//      (InputEvent.BUTTON3_DOWN_MASK|InputEvent.BUTTON1_DOWN_MASK)),InputEvent.BUTTON3_DOWN_MASK|InputEvent.BUTTON1_DOWN_MASK,InputEvent.BUTTON3_DOWN_MASK);
//  System.out.printf("click meta %o %o %o %o\n",e.getModifiersEx(),(e.getModifiersEx() & 
//      (InputEvent.META_DOWN_MASK|InputEvent.BUTTON1_DOWN_MASK)),InputEvent.META_DOWN_MASK|InputEvent.BUTTON1_DOWN_MASK,InputEvent.META_DOWN_MASK);
//  System.out.printf("click %o %o %o %o\n",e.getModifiers(),(e.getModifiers() & 
//      (InputEvent.BUTTON3_MASK|InputEvent.BUTTON1_DOWN_MASK)),InputEvent.BUTTON3_MASK|InputEvent.BUTTON1_DOWN_MASK,InputEvent.BUTTON3_MASK);
  if (operation==0)
    { // selezione multipla
    if ((e.getModifiersEx() & 
        (InputEvent.META_DOWN_MASK|InputEvent.BUTTON1_DOWN_MASK)) == InputEvent.META_DOWN_MASK)
//    if (e.isPopupTrigger())  Non funziona in Windows perché viene rilevato solo al mouseReleased
      {  // click col tasto destro
      if ((graphListeners.size()!=0 || usePopUp) && editing)
        {
        lastEvent=e;
        boolean selNode,selArc;
        NodeSign<TN> ns=getNode(e.getX(),e.getY());
        ArcSign<TN,TA> as=getArc(e.getX(),e.getY());
        selNode=ns!=null;
        selArc=as!=null;
        for (NodeSign<TN> nns:n)
          if (nns.selected)
            selNode=true;
        if (selNode)
          {
//          System.out.println("click su "+ns+" selNode="+selNode+" selected="+ns.selected);
//          ns.selected=!ns.selected;
          for (int i=0;i<graphListeners.size();i++)  // per attivare i listeners solo alla selezione interattiva
            graphListeners.elementAt(i).nodeSelected(new GraphEvent(this,ns));
          mnRemPU.setEnabled(true);
          mnRenPU.setEnabled(true);
          mnMinLPU.setEnabled(true);
          mnMinPPU.setEnabled(true);
//          mnSizePU.setEnabled(true); always enabled to change the default
//          mnImagePU.setEnabled(true); always enabled to change the default
          }
         else
          {
          mnRemPU.setEnabled(false);
          mnRenPU.setEnabled(false);
          mnMinLPU.setEnabled(false);
          mnMinPPU.setEnabled(false);
//          mnSizePU.setEnabled(false); // always enabled to change the default
//          mnImagePU.setEnabled(false); // always enabled to change the default
          }
        if (selArc)
          {
          for (int i=0;i<graphListeners.size();i++)  // per attivare i listeners solo alla selezione interattiva
            graphListeners.elementAt(i).arcSelected(new GraphEvent(this,as));
          mnUnlinkPU.setEnabled(true);
          mnWeightPU.setEnabled(true);
          }
         else
          {
          mnUnlinkPU.setEnabled(false);
//          mnWeightPU.setEnabled(false); // always enabled to change the default
          }
        if (selNode || selArc)
          {
          mnInfoPU.setEnabled(true);
          if ((e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) == (InputEvent.SHIFT_DOWN_MASK))
            dropSelected=false;
           else
            dropSelected=true;
//          System.out.println("dropSelected="+dropSelected);
          if (usePopUp)
            {
            mnPointPU.show(this,e.getX(),e.getY());
            }
          }
         else  // niente selezionato
          {
          mnWeightPU.setEnabled(true);
          mnRenPU.setEnabled(true);
          mnInfoPU.setEnabled(false);
          if (usePopUp)
            mnPointPU.show(this,e.getX(),e.getY());
          }
        }
       else if (editing)
        {
        removeNode(e.getX(),e.getY());
        removeArc(e.getX(),e.getY());
        }
      }
     else if (editing)  // tasto sinistro
      {
      if ((e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) == (InputEvent.SHIFT_DOWN_MASK))  // selezione multipla
        {
        boolean selNode,selArc;
        NodeSign<TN> ns=getNode(e.getX(),e.getY());
//        ArcSign<TN,TA> as=getArc(e.getX(),e.getY());
        selNode=ns!=null;
        if (selNode)
          {
//          System.out.println("click su "+ns+" selNode="+selNode+" selected="+ns.selected);
          ns.selected=!ns.selected;
          }
         else
          {
//          System.out.println("click su "+ns+" selNode="+selNode);
          }
        }
       else
        {  // inserisce i valori di default se l'immagine è null
        if (getNodeIndex(defaultName)<0)
          addNode(defaultName,dezoom(e.getX())-dx,dezoom(e.getY())-dy,defaultColor,defaultSize,defaultImage);
         else
          for (int nodeNum=0;true;nodeNum++)
            {
            if (getNodeIndex(defaultName+nodeNum)<0)
              {
              addNode(defaultName+nodeNum,dezoom(e.getX())-dx,dezoom(e.getY())-dy,defaultColor,defaultSize,defaultImage);
              break;
              }
            }
        }
      }
     else  // non modifica ma segnala la selezione
      {
      boolean selNode,selArc;
      NodeSign<TN> ns=getNode(e.getX(),e.getY());
      ArcSign<TN,TA> as=getArc(e.getX(),e.getY());
      selNode=ns!=null;
      selArc=as!=null;
      if (selNode)
        {
        for (int i=0;i<graphListeners.size();i++)  // per attivare i listeners solo alla selezione interattiva
          graphListeners.elementAt(i).nodeSelected(new GraphEvent(this,ns));
        }
      if (selArc)
        {
        for (int i=0;i<graphListeners.size();i++)  // per attivare i listeners solo alla selezione interattiva
          graphListeners.elementAt(i).arcSelected(new GraphEvent(this,as));
        }
      }
    }
   else
    {
    doOperation(e);
    }
//  System.out.println("finito mouseClicked "+defaultColor);
  repaint();
  }

/*****************
 * Esegue le diverse operazioni in seguito al click del mouse o al completamento di un comando popup
 * @param e l'evento da gestire
 *****************/
void doOperation(MouseEvent e)
  {
  String p[]=null;
  String nodeName=null;
  NodeSign<TN> ns;
  ArcSign<TN,TA> as;
  String nn;
  switch (operation)
    {
    case 0:
      break;
    case PSIZE:
      ns=getNode(e.getX(),e.getY());
      int oldSize;
      if (ns!=null)
        {
        oldSize=ns.getSize();
        nodeName=ns.getName();
        }
       else
        {
        oldSize=defaultSize;
        nodeName="";
        }
      nn=inputText(prompt[PSIZE]+(ns!=null?" "+prompt[SFOR]+" "+nodeName:""),oldSize+"",true);
      if (nn!=null)
        {
        defaultSize=parseInt(nn);
        resizeNode(nodeName,defaultSize);
        }
      for (int nr=0;nr<n.length;nr++)
        {
        if (n[nr].selected)
          {
          resizeNode(n[nr].getName(),defaultSize);
          if (dropSelected)
            {
            n[nr].selected=false;
            }
          }
        }
      e.consume();
      break;
    case PSHAPE:
      ns=getNode(e.getX(),e.getY());
      int oldShape;
      if (ns!=null)
        {
        oldShape=ns.getShape();
        nodeName=ns.getName();
        }
       else
        {
        oldShape=defaultShape;
        nodeName="";
        }
      String type=prompt[SCIRCLE];
      if (oldShape==NodeSign.CIRCLE)
        type=prompt[SCIRCLE];
       else if (oldShape==NodeSign.SQUARE)
        type=prompt[SSQUARE];
       else if (oldShape==NodeSign.CROSS)
        type=prompt[SCROSS];
       else
        type=""+oldShape;
      nn=inputText(prompt[PSHAPE]+(ns!=null?" "+prompt[SFOR]+" "+nodeName:""),type,true);
      if (nn!=null)
        {
        if (nn.equalsIgnoreCase(prompt[SCIRCLE]))
          defaultShape=NodeSign.CIRCLE;
         else if (nn.equalsIgnoreCase(prompt[SSQUARE]))
          defaultShape=NodeSign.SQUARE;
         else if (nn.equalsIgnoreCase(prompt[SCROSS]))
          defaultShape=NodeSign.CROSS;
         else defaultShape=parseInt(nn);
        reshapeNode(nodeName,defaultShape);
        }
      for (int nr=0;nr<n.length;nr++)
        {
        if (n[nr].selected)
          {
          reshapeNode(n[nr].getName(),defaultShape);
          if (dropSelected)
            {
            n[nr].selected=false;
            }
          }
        }
      e.consume();
      break;
    case PFILL:
      ns=getNode(e.getX(),e.getY());
      boolean oldFill;
      if (ns!=null)
        {
        oldFill=ns.isFilled();
        nodeName=ns.getName();
        }
       else
        {
        oldFill=defaultFill;
        nodeName="";
        }
      nn=inputText(prompt[PFILL]+(ns!=null?" "+prompt[SFOR]+" "+nodeName:""),(oldFill?prompt[STRUE]:prompt[SFALSE]),true);
      if (nn!=null)
        {
        if (nn.equalsIgnoreCase(prompt[SFALSE]) || nn.equalsIgnoreCase(prompt[SNO]))
          defaultFill=false;
         else if (nn.equalsIgnoreCase(prompt[STRUE]) || nn.equalsIgnoreCase(prompt[SYES]))
          defaultFill=true;
         else defaultFill=(parseInt(nn)==0)?false:true;
        changeNodeFill(nodeName,defaultFill);
        }
      for (int nr=0;nr<n.length;nr++)
        {
        if (n[nr].selected)
          {
          changeNodeFill(n[nr].getName(),defaultFill);
          if (dropSelected)
            {
            n[nr].selected=false;
            }
          }
        }
      e.consume();
      break;
    case PTHICK:
      ns=getNode(e.getX(),e.getY());
      int oldThickness;
      if (ns!=null)
        {
        oldThickness=ns.getThickness();
        nodeName=ns.getName();
        nn=inputText(prompt[PTHICKN]+" "+prompt[SFOR]+" "+nodeName,oldThickness+"",true);
        if (nn!=null)
          {
          defaultThickness=parseInt(nn);
          setThickness(nodeName,defaultThickness);
          }
        }
      for (int nr=0;nr<n.length;nr++)
        {
        if (n[nr].selected)
          {
          setThickness(n[nr].getName(),defaultThickness);
          if (dropSelected)
            {
            n[nr].selected=false;
            }
          }
        }
      as=getArc(e.getX(),e.getY());
      if (as!=null)
        {
        oldThickness=as.getThickness();
        nn=inputText(prompt[PTHICKA],oldThickness+"",true);
        if (nn!=null)
          {
          defaultThickness=parseInt(nn);
          as.setThickness(defaultThickness);
          setModified(true);
          }
        }
      if (as==null && ns==null)
        {
        oldThickness=defaultThickness;
        nn=inputText(prompt[PTHICKN],oldThickness+"",true);
        if (nn!=null)
          {
          defaultThickness=parseInt(nn);
          }
        }
      e.consume();
      break;
    case PFIX:
      ns=getNode(e.getX(),e.getY());
      boolean oldFix;
      if (ns!=null)
        {
        oldFix=ns.isFixed();
        nodeName=ns.getName();
        }
       else
        {
        oldFix=defaultFixing;
        nodeName="";
        }
      nn=inputText(prompt[PFIX]+(ns!=null?" "+prompt[SFOR]+" "+nodeName:""),(oldFix?prompt[STRUE]:prompt[SFALSE]),true);
      if (nn!=null)
        {
        if (nn.equalsIgnoreCase(prompt[SFALSE]) || nn.equalsIgnoreCase(prompt[SNO]))
          defaultFixing=false;
         else if (nn.equalsIgnoreCase(prompt[STRUE]) || nn.equalsIgnoreCase(prompt[SYES]))
          defaultFixing=true;
         else defaultFixing=(parseInt(nn)==0)?false:true;
        changeNodeFix(nodeName,defaultFixing);
        }
      for (int nr=0;nr<n.length;nr++)
        {
        if (n[nr].selected)
          {
          changeNodeFix(n[nr].getName(),defaultFixing);
          if (dropSelected)
            {
            n[nr].selected=false;
            }
          }
        }
      e.consume();
      break;
    case PIMAGE:
      ns=getNode(e.getX(),e.getY());
      String oldImage;
      if (ns!=null)
        {
        oldImage=(ns.getImageName()!=null)?ns.getImageName():"";
        nodeName=ns.getName();
        }
       else
        {
        oldImage=defaultImage;
        nodeName="";
        }
      nn=inputText(prompt[PIMAGE]+(ns!=null?" "+prompt[SFOR]+" "+nodeName:""),oldImage,true);
      if (nn!=null)
        {
        if (nn.equals(""))
          nn=null;
        defaultImage=nn;
        changeNodeImage(nodeName,defaultImage);
        }
      for (int nr=0;nr<n.length;nr++)
        {
        if (n[nr].selected)
          {
          changeNodeImage(n[nr].getName(),defaultImage);
          if (dropSelected)
            {
            n[nr].selected=false;
            }
          }
        }
      e.consume();
      break;
    case PREN:
      nodeName=getNodeName(e.getX(),e.getY());
      if (nodeName!=null)
        {
        nn=inputText(prompt[PREN1]+nodeName,nodeName,true);
        if (nn!=null)
          {
          defaultName=nn;
          renameNode(nodeName,nn);
          }
        }
       else
        {
        nn=inputText(prompt[PNAME],defaultName,true);
        if (nn!=null)
          {
          defaultName=nn;
          }
        }
      e.consume();
      break;
    case PMINL: case PMINP:
      first=getNodeName(e.getX(),e.getY());
      if (first!=null)
        {
        switch(operation)
          {
          case PMINL:
            operation=PMINL1;
            break;
          case PMINP:
            operation=PMINP1;
            break;
          }
        }
      e.consume();
      break;
    case PMINL1: case PMINP1:
      String second=getNodeName(e.getX(),e.getY());
//      System.out.println("first node="+first+", second="+second);
      p=null;
      switch (operation)
        {
        case PMINP1:
          p=g.minPricePath(first,second);
          break;
        case PMINL1:
          p=g.minLengthPath(first,second);
          break;
        }
//      for (int i=0;i<p.length;i++)
//        System.out.println(","+p[i]);
      if (p!=null)
        drawPath(defaultColor,p);
      operation=0;
      setModified(true);
      e.consume();
      break;
    case PCOLOR:
      ns=getNode(e.getX(),e.getY());
      defaultColor=inputColor();
      if (ns!=null)
        {
        ns.setColor(defaultColor);
        setModified(true);
        }
      as=getArc(e.getX(),e.getY());
      if (as!=null)
        {
        as.setColor(defaultColor);
        setModified(true);
        }
//      System.out.println("finito cambio colore "+defaultColor);
      for (int nr=0;nr<n.length;nr++)
        {
        if (n[nr].selected)
          {
          n[nr].setColor(defaultColor);
          if (dropSelected)
            {
            n[nr].selected=false;
            }
          setModified(true);
          }
        }
      e.consume();
      break;
    case PWEIGHT:
      ArcSign<TN,TA> as1=getArc(e.getX(),e.getY());
      try
        {
        if (as1!=null)
          nn=inputText(prompt[PWEIGHT],""+as1.a.p,true);
         else
          nn=inputText(prompt[PWEIGHT],""+defaultWeight,true);
        if (nn!=null)
          {
          defaultWeight=Double.parseDouble(nn.trim());
          if (as1!=null)
            {
            as1.a.p=defaultWeight;
            setModified(true);
            }
          }
        }
       catch (NumberFormatException nfe)
        {
        }
      e.consume();
      break;
    case PZOOM:
      try
        {
        nn=inputText(prompt[PZOOM],""+getZoom(),true);
        if (nn!=null)
          {
          setZoom(Integer.parseInt(nn.trim()));
          }
        }
       catch (NumberFormatException nfe)
        {
        }
      e.consume();
      break;
    case PINFO:
      ns=getNode(e.getX(),e.getY());
      if (ns!=null)
        {
        info=new Info<TN>(ns.getNode().getName(),ns.getNode().getValue(),e.getXOnScreen(),e.getYOnScreen());
        for (int nr=0;nr<n.length;nr++)
          {
          if (n[nr].selected)
            {
            if (dropSelected)
              {
              n[nr].selected=false;
              }
            }
          }
        }
      as=getArc(e.getX(),e.getY());
      if (info==null && as!=null)
        {
        Arc<TN,TA> ag=as.getArc();
        String w=""+ag.p;
        for (ag=ag.next;ag!=null;ag=ag.next)
          w+="|"+ag.p;
        info=new Info<TA>(w,as.getArc().getInfo(),e.getXOnScreen(),e.getYOnScreen());
        }
      e.consume();
      break;
    }
  if (operation!=PMINL1 && operation!=PMINP1)
    {
    operation=0;
    }
  repaint();
  }

/*************
 * Gestisce l'evento mouse entered
 * @param e l'evento del mouse
 *************/
public void mouseEntered(MouseEvent e)
  {
//  System.out.println("mouseentered dx="+dx+" dy="+dy+" oldx="+oldx+" oldy="+oldy);
  }

/*************
 * Gestisce l'evento mouse exited
 * @param e l'evento del mouse
 *************/
public void mouseExited(MouseEvent e)
  {
//  System.out.println("mouseexited dx="+dx+" dy="+dy+" oldx="+oldx+" oldy="+oldy);
  }

/*************
 * Indice del nodo selezionato
 *************/
int selectedNode=-1;
/*************
 * <b>true</b> se viene evidenziato l'elastico col punto precedentemente clickato, <b>false</b>
 *************/
boolean lineDraw=false;
/*************
 * ascissa del punto di partenza dell'elastico
 *************/
int linePosX;
/*************
 * ordinata del punto di partenza dell'elastico
 *************/
int linePosY;
/*************
 * <b>true</b> se viene evidenziato il rettangolo di selezione, <b>false</b>
 *************/
boolean multipleSelection=false;
/*************
 * ascissa del punto di partenza della selezione
 *************/
int startSelX;
/*************
 * ordinata del punto di partenza della selezione
 *************/
int startSelY;
/*************
 * ascissa del punto di arrivo della selezione
 *************/
int selPosX;
/*************
 * ordinata del punto di arrivo della selezione
 *************/
int selPosY;

/*************
 * Gestisce l'evento mouse pressed se l'evento non e' stato gia' consumato
 * @param e l'evento del mouse
 *************/
public void mousePressed(MouseEvent e)
  {
  if (info!=null)
    {
    info.setVisible(false);
    info.dispose();
    info=null;
    }
  if (sliding)
    {
    oldx=dezoom(e.getX())-dx;
    oldy=dezoom(e.getY())-dy;
    posx=e.getX();
    posy=e.getY();
    }
//  System.out.println("mousepressed dx="+dx+" dy="+dy+" oldx="+oldx+" oldy="+oldy);
  if (!editing && !movable || e.isConsumed())
    return;
//  System.out.println("Mouse pressed in GraphPanel");
//  System.out.format("%x %x\n",e.getModifiersEx(),InputEvent.SHIFT_DOWN_MASK);
  int i=getNodeIndex(e.getX(),e.getY());
  selectedNode=i;
  if (i>=0)
    {
//    System.out.println("pressed su "+n[selectedNode]+" selNode="+selectedNode+" selected="+n[selectedNode].selected);
    if ((e.getModifiersEx() &
        (InputEvent.BUTTON3_DOWN_MASK|InputEvent.BUTTON1_DOWN_MASK)) == (InputEvent.BUTTON3_DOWN_MASK))
      { // schiacciato col tasto destro
      linePosX=e.getX();
      linePosY=e.getY();
      }
    n[selectedNode].selected=!n[selectedNode].selected;
    }
   else if ((e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) == (InputEvent.SHIFT_DOWN_MASK))
    {
//    System.out.println("starting selection");
    startSelX=e.getX();
    startSelY=e.getY();
    }
  repaint();
  }

/*************
 * Gestisce l'evento mouse released se l'evento non e' stato gia' consumato
 * @param e l'evento del mouse
 *************/
public void mouseReleased(MouseEvent e)
  {
//  if (selectedNode>=0)
//    System.out.println("released su "+n[selectedNode]+" selNode="+selectedNode+" selected="+n[selectedNode].selected);
//  System.out.println("mousereleased dx="+dx+" dy="+dy+" oldx="+oldx+" oldy="+oldy);
  if (!editing && !movable || e.isConsumed())
//  if (!editing || e.isConsumed())
    return;
//  System.out.println("Mouse released in GraphPanel ");
  if (getNodeName(e.getX(),e.getY())!=null && lineDraw)
    addArc(n[selectedNode].nn.name,getNodeName(e.getX(),e.getY()));
  lineDraw=false;
  if (selectedNode!=-1)
    n[selectedNode].selected=!n[selectedNode].selected;
  selectedNode=-1;
  for (int nr=0;multipleSelection && nr<n.length;nr++)
    {
    int posX=zoom(dx)+zoom(n[nr].getX()),posY=zoom(dy)+zoom(n[nr].getY());
    if (((startSelX<posX && posX<selPosX)||(selPosX<posX && posX<startSelX)) &&
        ((startSelY<posY && posY<selPosY)||(selPosY<posY && posY<startSelY)))
      n[nr].selected=!n[nr].selected;
    }
  multipleSelection=false;
  repaint();
  }

/*************
 * Gestisce l'evento mouse dragged se l'evento non e' stato gia' consumato
 * @param e l'evento del mouse
 *************/
public void mouseDragged(MouseEvent e)
  {
//  System.out.println("mousedragged dx="+dx+" dy="+dy+" oldx="+oldx+" oldy="+oldy);
  if (!sliding && !movable || e.isConsumed())
    return;
//  System.out.printf("drag %o %o %o %o\n",e.getModifiersEx(),(e.getModifiersEx() & 
//      (InputEvent.BUTTON3_DOWN_MASK|InputEvent.BUTTON1_DOWN_MASK)),InputEvent.BUTTON3_DOWN_MASK|InputEvent.BUTTON1_DOWN_MASK,InputEvent.BUTTON3_DOWN_MASK);
//  System.out.printf("drag %o %o %o %o\n",e.getModifiers(),(e.getModifiers() & 
//      (InputEvent.BUTTON3_MASK|InputEvent.BUTTON1_DOWN_MASK)),InputEvent.BUTTON3_MASK|InputEvent.BUTTON1_DOWN_MASK,InputEvent.BUTTON3_MASK);
  if (selectedNode!=-1)
    {
    if (editing && (e.getModifiersEx() &
        (InputEvent.BUTTON3_DOWN_MASK|InputEvent.BUTTON1_DOWN_MASK)) == (InputEvent.BUTTON3_DOWN_MASK))
      { // col tasto destro viene segnato un nuovo arco
      lineDraw=true;
      linePosX=e.getX();
      linePosY=e.getY();
      }
     else if (movable)
//     else
      {
      int moveX=n[selectedNode].x-(dezoom(e.getX())-dx);
      int moveY=n[selectedNode].y-(dezoom(e.getY())-dy);
//      n[selectedNode].x=dezoom(e.getX())-dx;
//      n[selectedNode].y=dezoom(e.getY())-dy;
      for (int nr=0;nr<n.length;nr++)
        {
//        if (nr!=selectedNode && n[nr].selected)
        if (nr==selectedNode || n[nr].selected && !n[nr].fixed)
          {
          n[nr].setX(n[nr].getX()-moveX);
          n[nr].setY(n[nr].getY()-moveY);
          }
        }
      setModified(true);
      }
    }
   else
    {
    if ((e.getModifiersEx() &
        (InputEvent.BUTTON3_DOWN_MASK|InputEvent.BUTTON1_DOWN_MASK)) == (InputEvent.BUTTON3_DOWN_MASK))
      {   // cambia lo zoom di rappresentazione
      setZoom((posy-e.getY())/5);
      for (int i=0;i<graphListeners.size();i++)   // per attivare i listeners
        graphListeners.elementAt(i).zoomed(new GraphEvent(this,getZoom()));
//      System.out.print("x="+posx+","+e.getX()+" y="+posy+","+e.getY());
//      System.out.println(" zoom="+getZoom()+" diffx="+(posx-e.getX())+" diffy="+(posy-e.getY()));
      }
     else if ((e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) == (InputEvent.SHIFT_DOWN_MASK))
      {
      multipleSelection=true;
      selPosX=e.getX();
      selPosY=e.getY();
      }
     else
      {   // sposta la rappresentazione
      int newx=dezoom(e.getX()),newy=dezoom(e.getY());
      dx=(newx-oldx);
      dy=(newy-oldy);
      for (int i=0;i<graphListeners.size();i++)   // per attivare i listeners
        graphListeners.elementAt(i).moved(new GraphEvent(this,newx,newy));
//      System.out.println("mousedragged oldx="+oldx+",oldy="+oldy+",newx="+newx+",newy="+newy+"dx="+dx+",dy="+dy);
      }
    }
  repaint();
  }

/*************
 * Gestisce l'evento mouse moved
 * @param e l'evento del mouse
 *************/
public void mouseMoved(MouseEvent e)
  {
//  System.out.println("mousemoved dx="+dx+" dy="+dy+" oldx="+oldx+" oldy="+oldy);
  }

/*************
 * Ritorna il grafo rappresentato nel pannello
 * @return il grafo
 *************/
public Graph<TN,TA> getGraph()
  {
  return g;
  }

/*************
 * Ritorna l'ordine del grafo rappresentato nel GraphJPanel
 * @return il grado (ordine) del grafo
 *************/
public int degree()
  {
  return g.degree();
  }

/*************
 * Ritorna il nome di un nodo indicato con le sue coordinate sul grafo
 * @param x l'ascissa sul grafo
 * @param y l'ordinata sul grafo
 * @return il nome del nodo
 *************/
public String getNodeName(int x, int y)
  {
  int i=getNodeIndex(x,y);
  if (i<0)
    return null;
   else
    return n[i].nn.name;
  }

/*************
 * Ritorna un nodo indicato con il suo indice
 * @param i l'indice del nodo
 * @return l'indice del nodo, null se il nodo non esiste
 *************/
public NodeSign<TN> getNode(int i)
  {
  return (i<0 || i>=n.length)?null:n[i];
  }

/*************
 * Ritorna un nodo indicato con il suo nome
 * @param s il nome
 * @return l'indice del nodo, null se il nodo non esiste
 *************/
public NodeSign<TN> getNode(String s)
  {
  int i=getNodeIndex(s);
  return (i<0)?null:n[i];
  }

/*************
 * Ritorna un nodo indicato con le sue coordinate sul grafo
 * @param x l'ascissa sul grafo
 * @param y l'ordinata sul grafo
 * @return il nodo alla posizione indicata, null se non esiste alcun nodo in tale posizione
 *************/
public NodeSign<TN> getNode(int x, int y)
  {
  int i=getNodeIndex(x,y);
  if (i<0)
    return null;
   else
    return n[i];
  }

/*************
 * Ritorna l'indice di un nodo indicato con il suo nome
 * @param s il nome
 * @return l'indice del nodo, -1 se il nodo non esiste
 *************/
public int getNodeIndex(String s)
  {
  s=s.replace(" ","_");
  for (int i=0;i<n.length;i++)
    if (n[i].nn.name.equals(s))
      {
      return i;
      }
  return -1;
  }
  
/*************
 * Ritorna l'indice di un nodo indicato con le sue coordinate sul grafo
 * @param x l'ascissa sul grafo
 * @param y l'ordinata sul grafo
 * @return l'indice del nodo, -1 se il nodo non esiste
 *************/
public int getNodeIndex(int x, int y)
  {
  for (int i=0;i<n.length;i++)
    if (Math.abs(dezoom(x)-dx-n[i].x)<n[i].dim &&
        Math.abs(dezoom(y)-dy-n[i].y)<n[i].dim)
      {
      return i;
      }
  return -1;
  }

/*************
 * Ritorna la matrice dei nodi presenti nel GraphPanel
 * @return la matrice dei nodi
 *************/
public NodeSign<TN>[] getNodes()
  {
  return n;
  }

/*************
 * Ritorna l'arco tra due nodi, se presente
 * @param n1 il nodo di partenza
 * @param n2 il nodo di arrivo
 * @return l'arco tra i due nodi, null se non esiste alcun arco
 *************/
public ArcSign<TN,TA> getArc(NodeSign<TN> n1, NodeSign<TN> n2)
  {
  return getArc(n1.getName(),n2.getName());
  }

/*************
 * Ritorna l'arco tra due nodi, se presente
 * @param n1 il nodo di partenza
 * @param n2 il nodo di arrivo
 * @return l'arco tra i due nodi, null se non esiste alcun arco
 *************/
public ArcSign<TN,TA> getArc(String n1, String n2)
  {
  int ni1=getNodeIndex(n1);
  int ni2=getNodeIndex(n2);
  if (ni1<0 || ni2<0)
    return null;
   else
    return a[ni1][ni2];
  }

/*************
 * Ritorna l'arco indicato con le sue coordinate sul grafo
 * @param x l'ascissa sul grafo
 * @param y l'ordinata sul grafo
 * @return l'arco alla posizione indicata, null se non esiste alcun arco in tale posizione
 *************/
public ArcSign<TN,TA> getArc(int x,int y)
  {
  for (int i=0;i<a.length;i++)
    for (int j=0;j<a[i].length;j++)
      {
      if (a[i][j]!=null)
        if (i!=j)
          {
          NodeSign<TN> f=a[i][j].from;
          NodeSign<TN> t=a[i][j].to;
          double x0=f.x,y0=f.y,
              x1=t.x,y1=t.y,
              xp=dezoom(x)-dx,yp=dezoom(y)-dy;
          if ((((x1-2*xp+x0)*(x1-2*xp+x0)/4)+((y1-2*yp+y0)*(y1-2*yp+y0)/4))-
              (((x1-x0)/2)*((x1-x0)/2)+((y1-y0)/2)*((y1-y0)/2))<0 &&
              (Math.abs((y1-y0)*(xp-x0)-(x1-x0)*(yp-y0))<10*f.dim*t.dim))
            return a[i][j];
          }
         else
          {
          NodeSign<TN> f=a[i][j].from;
          int xp=dezoom(x)-dx,yp=dezoom(y)-dy;
          if (Math.abs(xp-f.x-f.dim)<f.dim &&
              Math.abs(yp-f.y-f.dim)<f.dim)
            return a[i][j];
          }
      }
  return null;
  }

/*************
 * Ritorna la matrice degli archi presenti nel GraphPanel
 * @return la matrice degli archi
 *************/
public ArcSign<TN,TA>[][] getArcs()
  {
  return a;
  }

/*************
 * Stabilisce se visualizzare gli assi nel grafo
 * @param show true se gli assi vanno visualizzati, false altrimenti
 *************/
public void setShowAxis(boolean show)
  {
  showAxis=show;
  repaint();
  }
  
/*************
 * Ritorna la visualizzazione degli assi del grafo
 * @return true se gli assi vanno visualizzati, false altrimenti
 *************/
public boolean getShowAxis()
  {
  return showAxis;
  }
  
/*************
 * Stabilisce se visualizzare i nomi dei nodi nel grafo
 * @param show true se le scritte vanno visualizzate, false altrimenti
 *************/
public void setShowNames(boolean show)
  {
  if (showNames!=show)
    notifySetting();
  showNames=show;
  repaint();
  }
  
/*************
 * Ritorna la visualizzazione dei nomi dei nodi del grafo
 * @return true se le scritte vanno visualizzate, false altrimenti
 *************/
public boolean getShowNames()
  {
  return showNames;
  }
  
/*************
 * Stabilisce se visualizzare i pesi degli archi nel grafo
 * @param show true se le scritte vanno visualizzate, false altrimenti
 *************/
public void setShowWeights(boolean show)
  {
  if (showWeights!=show)
    notifySetting();
  showWeights=show;
  repaint();
  }
  
/*************
 * Ritorna la visualizzazione dei pesi degli archi del grafo
 * @return true se le scritte vanno visualizzate, false altrimenti
 *************/
public boolean getShowWeights()
  {
  return showWeights;
  }
  
/*************
 * Stabilisce se visualizzare le scritte nel grafo
 * @param show true se le scritte vanno visualizzate, false altrimenti
 *************/
public void setShowText(boolean show)
  {
  if (showNames!=show || showWeights!=show)
    notifySetting();
  showNames=show;
  showWeights=show;
  repaint();
  }
  
/*************
 * Stabilisce la stringa di default
 * @param s la stringa di default
 + @deprecated substituted by setName
 *************/
@Deprecated
public void setString(String s)
  {
  setName(s);
  }
  
/*************
 * Stabilisce il nome di default
 * @param s la stringa di default
 *************/
public void setName(String s)
  {
  defaultName=s;
  notifySetting();
  }
  
/*************
 * Ritorna la stringa di default
 * @return la stringa di default
 * @deprecated substituted by getName
 *************/
@Deprecated
public String getString()
  {
  return getName();
  }

/*************
 * Ritorna il nome di default
 * @return il nome di default
 *************/
public String getName()
  {
  return defaultName;
  }

/*************
 * Stabilisce il peso di default
 * @param w il peso di default
 *************/
public void setWeight(double w)
  {
  defaultWeight=w;
  notifySetting();
  }
  
/*************
 * Ritorna il peso di default
 * @return il peso di default
 *************/
public double getWeight()
  {
  return defaultWeight;
  }
  
/*************
 * Stabilisce il colore di default
 * @param c il colore di default
 *************/
public void setColor(Color c)
  {
  defaultColor=c;
  notifySetting();
  }
  
/*************
 * Ritorna il colore di default
 * @return il colore di default
 *************/
public Color getColor()
  {
  return defaultColor;
  }
  
/*************
 * Stabilisce lo spessore di default
 * @param th lo spessore di default
 *************/
public void setThickness(int th)
  {
  defaultThickness=th;
  notifySetting();
  }
  
/*************
 * Ritorna lo spessore di default
 * @return lo spessore di default
 *************/
public int getThickness()
  {
  return defaultThickness;
  }
  
/*************
 * Stabilisce la dimensione dei nodi di default
 * @param dim la dimensione dei nodi di default
 *************/
public void setNodeSize(int dim)
  {
  defaultSize=dim;
  notifySetting();
  }
  
/*************
 * Ritorna la dimensione dei nodi di default
 * @return la dimensione dei nodi default
 *************/
public int getNodeSize()
  {
  return defaultSize;
  }
  
/*************
 * Stabilisce la forma dei nodi di default
 * @param sh la forma dei nodi di default
 *************/
public void setNodeShape(int sh)
  {
  defaultShape=sh;
  notifySetting();
  }
  
/*************
 * Ritorna la forma dei nodi di default
 * @return la forma dei nodi default
 *************/
public int getNodeShape()
  {
  return defaultShape;
  }
  
/*************
 * Stabilisce il rempimento di dafault dei nodi
 * @param fill il riempimento di default dei nodi
 *************/
public void setNodeFill(boolean fill)
  {
  defaultFill=fill;
  notifySetting();
  }
  
/*************
 * Ritorna il riempimento di default dei nodi
 * @return la riempimento di default dei nodi
 *************/
public boolean getNodeFill()
  {
  return defaultFill;
  }
  
/*************
 * Stabilisce il fissaggio di dafault dei nodi
 * @param fixing il fissaggio di default dei nodi
 *************/
public void setNodeFixing(boolean fixing)
  {
  defaultFixing=fixing;
  notifySetting();
  }
  
/*************
 * Ritorna il fissaggio di default dei nodi
 * @return la fissaggio di default dei nodi
 *************/
public boolean getNodeFixing()
  {
  return defaultFixing;
  }
  
/*************
 * Stabilisce l'immagine dei nodi di default
 * @param im l'immagine dei nodi di default
 *************/
public void setImageName(String im)
  {
  defaultImage=im;
  notifySetting();
  }
  
/*************
 * Ritorna l'immagine dei nodi di default
 * @return l'immagine dei nodi di default
 *************/
public String getImageName()
  {
  return defaultImage;
  }
  
/*************
 * Ingrandisce il grafo
 *************/
public void inZoom()
  {
  dx=zoom(dx);
  dy=zoom(dy);
  if (zoomRate<MAXZOOM)
    zoomRate++;
//  defaultSize=20+5*zoomRate;
  dx=dezoom(dx);
  dy=dezoom(dy);
  for (int i=0;i<graphListeners.size();i++)   // per attivare i listeners
    graphListeners.elementAt(i).zoomed(new GraphEvent(this,getZoom()));
  repaint();
  }

/*************
 * Rimpicciolisce il grafo
 *************/
public void outZoom()
  {
  dx=zoom(dx);
  dy=zoom(dy);
  if (zoomRate>MINZOOM)
    zoomRate--;
//  defaultSize=20-5*zoomRate;
  dx=dezoom(dx);
  dy=dezoom(dy);
  for (int i=0;i<graphListeners.size();i++)   // per attivare i listeners
    graphListeners.elementAt(i).zoomed(new GraphEvent(this,getZoom()));
  repaint();
  }

/*************
 * Stabilisce il grado di ingrandimento del grafo
 * @param z il grado di ingrandimento del grafo compreso tra MINZOOM (-2) e MAXZOOM (7)
 *************/
public void setZoom(int z)
  {
  zoomRate=(z<MINZOOM?MINZOOM:z>MAXZOOM?MAXZOOM:z);
  for (int i=0;i<graphListeners.size();i++)   // per attivare i listeners
    graphListeners.elementAt(i).zoomed(new GraphEvent(this,getZoom()));
  repaint();
  }

/*************
 * Ritorna il grado di ingrandimento del grafo
 * @return il grado di ingrandimento del grafo
 *************/
public int getZoom()
  {
  return zoomRate;
  }

/*************
 * Rimuove la zoommata dalla dimensione
 * @param v il valore da dezoommare
 * @return la dimensione alle coordinate assolute
 *************/
public int dezoom(int v)
  {
  return (int)(4.0*v/(4.0+zoomRate));
  }

/*************
 * Applica la zoommata alla dimensione
 * @param v il valore da zoommare
 * @return la dimensione all'ingrandimento attuale
 *************/
public int zoom(int v)
  {
  return (int)(v*(1+zoomRate/4.0));
  }

/*************
 * Ripristina la zoommata e l'origine
 *************/
public void reset()
  {
  dx=0;
  dy=0;
  setZoom(0);
  repaint();
  }

/*************
 * Sposta l'origine
 * @param x lo spostamento orizzontale
 * @param y lo spostamento verticale
 *************/
public void slide(int x, int y)
  {
  dx-=x;
  dy-=y;
  repaint();
  }

/************
 * Calcola l'ascissa assoluta dato un punto del piano
 * @param x l'ascissa del piano
 * @return l'ascissa assoluta
 ************/
public int absoluteX(int x)
  {
  return dezoom(x)-dx;
  }

/************
 * Calcola l'ordinata assoluta dato un punto del piano
 * @param y l'ordinata del piano
 * @return l'ordinata assoluta
 ************/
public int absoluteY(int y)
  {
  return dezoom(y)-dy;
  }

/*************
 * Stabilisce l'ImageReader da utilizzare per la lettura delle immagini
 * @param i l'ImageReader da utilizzare
 *************/
public void setImageReader(ImageReader i)
  {
  ir=i;
  }

/*************
 * Ritorna il nome dello sfondo 
 * @return il nome del file contenente l'immagine di sfondo
 *************/
public String getBackImage()
  {
  return bgName;
  }

/*************
 * Ritorna l'immagine dello sfondo 
 * @return l'immagine di sfondo
 *************/
public Image getBackImageSource()
  {
  return bgImage;
  }

/*************
 * Modifica lo sfondo 
 * @param i il nome del file contenente l'immagine di sfondo
 *************/
public void setBackImage(String i)
  {
  if (i!=null && !i.equals(getBackImage()))
    {
    bgImage=ir.getImage(i);
    setModified(true);
    }
   else if (i==null && getBackImage()!=null)
    setModified(true);
  bgName=i;
  if (bgName==null)
    bgImage=null;
  notifySetting();
  repaint();
  }

/*************
 * Modifica lo sfondo 
 * @param i l'immagine di sfondo
 * @deprecated versione 2.0 utilizzare la forma setBackImage(String i) che conserva il nome dell'immagine per poterlo salvare
 *************/
@Deprecated
public void setBackImage(Image i)
  {
  bgImage=i;
  bgName=null;
  setModified(true);
  notifySetting();
  repaint();
  }

/*************
 * Ritorna una immagine. Implementa ImageReader
 * @param ims nome dell'immagine da ritornare
 * @return l'immagine letta
 *************/
public Image getImage(String ims)
  {
  Image im=null;
  try
    {
    if (new File(ims).exists())
      {
      im=Toolkit.getDefaultToolkit().getImage(ims);
      }
    }
   catch (AccessControlException ace)
    {
//    ace.printStackTrace();
    }
  if (im==null)
    {
    URL imUrl=getClass().getResource(ims);
//    System.out.println("Image "+ims+" in JAR is "+imUrl);
    if (imUrl!=null)
      im=Toolkit.getDefaultToolkit().getImage(imUrl);
    }
  if (im!=null)
    {
    MediaTracker mt=new MediaTracker(this);
    mt.addImage(im,0);
    try
      {
      mt.waitForAll();
      }
     catch(InterruptedException ie)
      {}
    }
  return im;
  }

/*************
 * Aggiunge un nuovo GraphListener al pannello.
 * @param gl il nuovo GraphListener da aggiungere
 *************/
public void addGraphListener(GraphListener gl)
  {
  graphListeners.add(gl);
  }
 
/*************
 * Rimuove un nuovo GraphListener dal pannello.
 * @param gl il nuovo GraphListener da aggiungere
 *************/
public void removeGraphListener(GraphListener gl)
  {
  graphListeners.removeElement(gl);
  }
 
/*************
 * Rimuove un GraphListener dal pannello.
 * @return la matrice contenente i GraphListener aggiunti al pannello
 *************/
public GraphListener[] getGraphListeners()
  {
  GraphListener listeners[]=new GraphListener[graphListeners.size()];
  for (int i=0;i<graphListeners.size();i++)
    listeners[i]=graphListeners.elementAt(i);
  return listeners;
  }
/*************
 * Disegna il GraphJPanel 
 * @param g lo spazio grafico di tracciamento
 *************/
public void update(Graphics g)
  {
  BufferedImage buffer=null;
  Graphics gbuf=null;
  buffer = new BufferedImage(getWidth(),getHeight(),BufferedImage.TYPE_INT_RGB);
  gbuf = buffer.getGraphics();
  paint(gbuf);
  g.drawImage(buffer,0,0,this);
  }

/*************
 * Disegna il pannello 
 * @param g lo spazio grafico di tracciamento
 *************/
public void paint(Graphics g)
  {
  g.setFont(new Font(null,Font.PLAIN,10+zoomRate));
  g.setColor(getBackground());
  if (bgImage!=null)
    {
    if (zoom(dx)>0 || zoom(dy)>0 ||
        zoom(dx+bgImage.getWidth(this))<getSize().width ||
        zoom(dy+bgImage.getHeight(this))<getSize().height)
      {
      g.fillRect(0,0,getSize().width,getSize().height);
      }
    g.drawImage(bgImage,zoom(dx),zoom(dy),
                        zoom(bgImage.getWidth(this)),zoom(bgImage.getHeight(this)),this);
    }
   else
    g.fillRect(0,0,getSize().width,getSize().height);
  /* draws axis */
  if (showAxis)
	  {
	  g.setColor(Color.black);
	  g.drawLine(zoom(dx),0,zoom(dx),getSize().height);
	  g.drawLine(0,zoom(dy),getSize().width,zoom(dy));
	  }
  for (int i=0;i<n.length;i++)
    {
    n[i].paint(g,dx,dy,1+zoomRate/4.0,showNames);
    for (int j=0;j<n.length;j++)
      {
//      for (ArcSign<TN,TA> asp=a[i][j];asp!=null;asp=asp.next)
//        System.out.println(asp);
      if (a[i][j]!=null)
        {
        a[i][j].paint(g,dx,dy,1+zoomRate/4.0,showWeights);
        }
      }
    }
  if (lineDraw)
    {
    g.setColor(Color.black);
    g.drawLine(zoom(dx)+zoom(n[selectedNode].x),zoom(dy)+zoom(n[selectedNode].y),linePosX,linePosY);
    }
  if (multipleSelection)
    {
    g.setColor(Color.black);
    int xSel,dxSel,ySel,dySel;
    if (startSelX>selPosX)
      {
      xSel=selPosX;
      dxSel=startSelX-selPosX;
      }
     else
      {
      xSel=startSelX;
      dxSel=selPosX-startSelX;
      }
    if (startSelY>selPosY)
      {
      ySel=selPosY;
      dySel=startSelY-selPosY;
      }
     else
      {
      ySel=startSelY;
      dySel=selPosY-startSelY;
      }
    g.drawRect(xSel,ySel,dxSel,dySel);
    }
  }
/************
 * Ritorna la stringa che descrive il grafo
 * @return la descrizione del grafo
 ************/
public String toString()
  {
  if (g!=null)
    return g.toString();
   else
    return null;
  }

@Override
public void setVisible(boolean v)
  {
  super.setVisible(v);
  if (!v && info!=null)
    {
    info.setVisible(false);
    info.dispose();
    info=null;
    }
  }


@SuppressWarnings({"rawtypes","unchecked","serial"})
class Info<T> extends JFrame
{
public Info(String label,T obj,int x,int y)
  {
  int nRows=2;
  int len=0;
  setUndecorated(true);
  getContentPane().removeAll();
  String sVal=label;
/*
  Field fld[]=null;
  if (obj!=null)
    {
    Class c=obj.getClass();
    fld=c.getDeclaredFields();
    nRows+=fld.length;
    for (int f=0;f<fld.length;f++)
      {
      String sf=fld[f].getName()+":"+fld[f].getGenericType().getTypeName()+"="+fld[f].toString();
                                        // ^ qui bisognerebbe mettere il valore del campo
      if (len<sf.length())
        len=sf.length();
      sVal+="\n"+sf;
      }
    }
*/
//  System.out.println("<"+sVal+">");
  if (obj!=null)
    {
    sVal+=":"+obj;
  //  getContentPane().add(new JLabel(sVal));
    getContentPane().setLayout(new GridLayout(nRows,1));
    }
//  System.out.println("<"+sVal+">");
  len=0;
  Scanner sclin=new Scanner(sVal);
  for (;sclin.hasNextLine();)
    {
    String lin=sclin.nextLine();
    if (lin.length()>len)
      len=lin.length();
//    int nP=ts.getFontMetrics(ts.getFont()).stringWidth(lin);
//    if (nP>len)
//      len=nP;
    nRows++;
//    System.out.println("<"+lin+"> "+nRows);
    }
  JTextArea ts=new JTextArea(sVal,nRows,len);
  ts.setEditable(false);
  getContentPane().add(ts);
//  getContentPane().add(new JScrollPane(ts));
/*
  System.out.println(ts);
  System.out.println(ts.getFont());
  System.out.println(ts.getFontMetrics(ts.getFont()));
  System.out.println(ts.getFontMetrics(ts.getFont()).stringWidth(sVal));
*/
//  System.out.println(""+String.format("%"+len+"s"," "));
  setSize(30+ts.getFontMetrics(ts.getFont()).stringWidth(sVal),ts.getFontMetrics(ts.getFont()).getHeight());
//  setSize(len,ts.getFontMetrics(ts.getFont()).getHeight()*nRows);
  setSize(30+ts.getFontMetrics(ts.getFont()).stringWidth(String.format("%"+len+"s","_")),ts.getFontMetrics(ts.getFont()).getHeight());
//  pack();
  validate();
  setLocation(x,y);
  setVisible(true);
  }
}
  
/*************
 * Il frame che contiene il pannello di prova
 *************/
static Frame f;

/*************
 * Il programma di prova
 * @param s i nomi dei file contenenti le descrizioni dei grafi. Se non ne sono indicati viene utilizzato un grafo di default
 *************/
public static void main(String s[])
  {
  if (s.length==0)
    {
    String n[]={"A","B","C","D","E"};
    int v[][]={{1,3,0,1,0}
              ,{0,0,1,1,0}
              ,{0,1,0,0,1}
              ,{0,0,1,0,0}
              ,{2,0,0,0,0}
              };
    double p[][][]={{{12.37},{3.2,7.4,6.9},{0},{106.1},{0}}
                 ,{{0},{0},{7.1},{2.3},{0}}
                 ,{{0},{0},{0},{0},{3}}
                 ,{{0},{0},{4.211},{0},{0}}
                 ,{{0.73,10.3},{0},{0},{0},{0}}
                 };
    f=new Frame("default graph");
    final GraphJPanel<Object,Object> gp;
    f.add(gp=new GraphJPanel<Object,Object>(n,v,p),BorderLayout.CENTER);
    f.addWindowListener(new WindowAdapter()
         {
         public void windowClosing(WindowEvent we)
           {
           if (gp!=null)
             gp.close();
           f.setVisible(false);
           f.dispose();
           }
         });
    f.setSize(200,200);
    JButton go=new JButton("Adding node F");
    f.add(go,BorderLayout.SOUTH);
    JButtonChanger bc=new JButtonChanger(go,gp);
    go.addActionListener(bc);
    f.pack();
    f.setVisible(true);
    }
   else
    {
    for (int i=0;i<s.length;i++)
      {
      try
        {
        f=new Frame(s[i]+" graph");
        final GraphJPanel<Object,Object> gp;
        f.add(gp=new GraphJPanel<Object,Object>(new FileReader(s[i])),BorderLayout.CENTER);
        f.addWindowListener(new WindowAdapter()
             {
             public void windowClosing(WindowEvent we)
               {
               if (gp!=null)
                 gp.close();
               f.setVisible(false);
               f.dispose();
               }
             });
        System.out.println(gp);
        System.out.println(gp.g);
        System.out.println(gp.g.connections());
        f.setSize(200,200);
        JButton go=new JButton("Va avanti");
        f.add(go,BorderLayout.SOUTH);
        go.addActionListener(new JButtonChanger(go,gp));
        f.pack();
        f.setVisible(true);
        }
       catch(FileNotFoundException fnfe)
        {
        f.setTitle("Missing file "+s[i]);
        }
      }
    }
  }
}

/*************
 * Il listener che  controlla il bottone del GraphPanel
 *************/
class JButtonChanger implements ActionListener
{
/*************
 * Il GraphPanel su cui agisce il bottone 
 *************/
@SuppressWarnings({"rawtypes","unchecked"})
GraphJPanel gp;
/*************
 * Il bottone da gestire 
 *************/
JButton f;
/*************
 * lo stato in cui si trova il GraphPanel di prova 
 *************/
int stato=0;
/*************
 * Crea un JButtonChanger
 * @param f il bottone da controllare
 * @param gp il GraphPanel associato
 *************/
@SuppressWarnings({"rawtypes","unchecked"})
public JButtonChanger(JButton f,GraphJPanel gp)
  {
  this.gp=gp;
  this.f=f;
  }

/*************
 * modifica il testo del bottone
 * @param b il bottone di cui cambiare il testo
 * @param s il testo da inserire
 *************/
void setText(JButton b,String s)
  {
  b.setText(s);
  }
/*************
 * Il gestore dell'evento
 * @param ae l'evento da gestire 
 *************/
@SuppressWarnings({"rawtypes","unchecked"})
public void actionPerformed(ActionEvent ae)
  {
  String voiceEn[]={"",
                    "Delete node",
                    "Delete arc",
                    "Min price -> other node",
                    "Min length -> other node",
                    "Change name",
                    "Change color",
                    "Change weight",
                    "Change zoom",
                    "Change size",
                    "Change image",
                    "Change shape",
                    "Change filling",
                    "Change fixing",
                  };
  String voiceIt[]={"",
                    "Cancella nodo",
                    "Cancella arco",
                    "Min prezzo -> altro nodo",
                    "Min lung -> altro nodo",
                    "Cambia nome",
                    "Cambia colore",
                    "Cambia peso",
                    "Cambia zoom",
                    "Cambia dimensione",
                    "Cambia immagine",
                    "Cambia forma",
                    "Cambia riempimento",
                    "Cambia fissaggio",
                   };
  String promptEn[]={"",
                     "weight of arcs:",
                     "new name for ",
                     "node name:",
                     "zoom (-2..7):",
                     "choose a color",
                     "new size for ",
                     "new image for ",
                     "new shape for ",
                     "set filling of ",
                  };
  String promptIt[]={"",
                     "peso dell'arco:",
                     "nuovo nome per ",
                     "nome nodo:",
                     "zoom (-2..7):",
                     "scegli colore",
                     "nuova dimensione per ",
                     "nuova immagine per ",
                     "nuova forma per ",
                     "scegli riempimento di ",
                    };
  switch(stato)
    {
    case 0:
      System.out.println("Adding node F");
      setText(f,"Adding objects to nodes");
      gp.addNode("F");
      System.out.println(gp);
      System.out.println(gp.g.connections());
      stato=22;
      break;
    case 22:
      System.out.println("Adding objects to nodes");
      setText(f,"Setting adaptive graph");
      gp.addNode("Info1\nPunto",40,40);
      Node n1=gp.getNode("Info1\nPunto").getNode();
      n1.setValue(new Point(40,40));
      gp.addNode("Info2\nGrafo",60,140);
      Node n2=gp.getNode("Info2\nGrafo").getNode();
      n2.setValue(gp.g);
      gp.addArc("Info1\nPunto","Info2\nGrafo");
      Arc a=gp.getArc("Info1\nPunto","Info2\nGrafo").getArc();
      a.setInfo(a);
      stato=21;
      break;
    case 21:
      System.out.println("Setting adaptive graph");
      setText(f,"Drawing vertical arcs between six squares");
      gp.setAdaptive(true);
      stato=20;
      break;
    case 20:
      System.out.println("Drawing vertical arcs between six squares");
      setText(f,"Setting adaptive false movable false and editable true");
      gp.addNode("upper",30,30,Color.blue,30,NodeSign.SQUARE);
      gp.addNode("lower",30,100,Color.blue,10,NodeSign.SQUARE);
      gp.addArc("lower","upper",Color.red,5,true,12.0);
      gp.addNode("upper1",120,30,Color.blue,30,NodeSign.SQUARE);
      gp.addNode("lower1",120,100,Color.blue,10,NodeSign.SQUARE);
      gp.addArc("upper1","lower1",Color.red,5,true,12.0);
      gp.addNode("left",180,30,Color.blue,30,NodeSign.SQUARE);
      gp.addNode("right",330,30,Color.blue,10,NodeSign.SQUARE);
      gp.addArc("left","right",Color.red,5,true,12.0);
      System.out.println(gp);
      stato=19;
      break;
    case 19:
      System.out.println("Setting adaptive false movable true and editable false");
      setText(f,"Setting movable false and editable true");
      gp.setAdaptive(false);
      gp.setMovable(true);
      gp.setEdit(false);
      stato=18;
      break;
    case 18:
      System.out.println("Setting movable false and editable true");
      setText(f,"Adding arc A-F PU false movable");
      gp.setMovable(false);
      gp.setEdit(true);
      stato=1;
      break;
    case 1:
      System.out.println("Adding arc A-F PU false movable");
      setText(f,"Drawing path F-C-D");
      gp.setMovable(true);
      gp.setPopUp(false);
      gp.addArc("A","F");
      System.out.println(gp);
      stato++;
      break;
    case 2:
      System.out.println("Drawing path F-C-D");
      setText(f,"Finding min path A-E");
      gp.drawPath(Color.red,new String[]{"F","C","D"});
      gp.setPopUp(true);
      stato++;
      break;
    case 3:
      System.out.println("Finding min path A-E");
      setText(f,"Finding short path A-E");
      String s[]=gp.g.minPricePath("A","E");
      System.out.println(gp);
      if (s!=null)
        {
        System.out.print("minPricePath ");
        for (int i=0;i<s.length;i++)
          System.out.print("\""+s[i]+"\":");
        System.out.println();
        gp.drawPath(Color.blue,s);
        }
       else
        System.out.println("minPricePath not connected");
      stato++;
      break;
    case 4:
      System.out.println("Finding short path A-E");
      setText(f,"Adding arc F-E 61.5");
      String s1[]=gp.g.minLengthPath("A","E");
      if (s1!=null)
        {
        System.out.print("minLengthPath ");
        for (int i=0;i<s1.length;i++)
          System.out.print("\""+s1[i]+"\":");
        System.out.println();
        gp.drawPath(Color.green,s1);
        }
       else
        System.out.println("minLengthPath not connected");
      stato++;
      break;
    case 5:
      System.out.println("Adding arc F-E 61.5");
      setText(f,"Finding min path A-E");
      gp.addArc("F","E",61.5);
      System.out.println(gp);
      stato++;
      break;
    case 6:
      System.out.println("Finding min path A-E");
      setText(f,"Finding short path A-E - italiano");
      String s2[]=gp.g.minPricePath("A","E");
      if (s2!=null)
        {
        System.out.print("minPricePath ");
        for (int i=0;i<s2.length;i++)
          System.out.print("\""+s2[i]+"\":");
        System.out.println();
        gp.drawPath(Color.pink,s2);
        }
       else
        System.out.println("minPricePath not connected");
      stato++;
      break;
    case 7:
      System.out.println("Finding short path A-E - italiano");
      gp.setLanguage(voiceIt,promptIt);
      setText(f,"Adding arc F E -1.5");
      String s3[]=gp.g.minLengthPath("A","E");
      if (s3!=null)
        {
        System.out.print("minLengthPath ");
        for (int i=0;i<s3.length;i++)
          System.out.print("\""+s3[i]+"\":");
        System.out.println();
        gp.drawPath(Color.cyan,s3);
        }
       else
        System.out.println("minLengthPath not connected");
      stato++;
      break;
    case 8:
      System.out.println("Adding arc F E -1.5");
      setText(f,"Finding min path A-E");
      gp.addArc("F","E",-1.5);
      stato++;
      break;
    case 9:
      System.out.println("Finding min path A-E");
      setText(f,"Remove arc A-B - english");
      String s4[]=gp.g.minPricePath("A","E");
      if (s4!=null)
        {
        System.out.print("minPricePath ");
        for (int i=0;i<s4.length;i++)
          System.out.print("\""+s4[i]+"\":");
        System.out.println();
        gp.drawPath(Color.yellow,s4);
        }
       else
        System.out.println("minLengthPath not connected");
      stato++;
      break;
    case 10:
      System.out.println("Remove arc A-B - english");
      gp.setLanguage(voiceEn,promptEn);
      setText(f,"Removing node C");
      gp.removeArc("A","B");
      System.out.println(gp);
      stato++;
      break;
    case 11:
      System.out.println("Removing node C");
      setText(f,"Adding node C with image");
      gp.removeNode("C");
      System.out.println(gp);
      stato++;
      break;
    case 12:
      System.out.println("Adding node C with image");
      setText(f,"Adding node M with filled square");
      gp.addNode("C",120,100,Color.pink,30,"Nodo.GIF");
      System.out.println(gp);
      stato++;
      break;
    case 13:
      System.out.println("Adding node M with filled square");
      setText(f,"Adding node N with filled circle");
      gp.addNode("M",20,40,new Color(0xff00fe),30,1,true);
      System.out.println(gp);
      stato++;
      break;
    case 14:
      System.out.println("Adding node N with filled circle");
      setText(f,"Adding path M N A");
      gp.addNode("N",70,80,new Color(0xfcf0fe),20,0,true);
      System.out.println(gp);
      stato++;
      break;
    case 15:
      System.out.println("Adding path M N A");
      setText(f,"Adding background image");
      gp.drawPath(new Color(0x0F000F),new String[]{"M","N","A"});
      System.out.println(gp);
      stato++;
      break;
    case 16:
      System.out.println("Adding background image");
      setText(f,"Linearising graph");
      gp.setBackImage("italia/Italia.gif");
      System.out.println(gp);
      stato++;
      break;
    case 17:
      System.out.println("Linearising");
      setText(f,"Bye");
      gp.linearise();
      System.out.println(gp);
      System.out.println(gp.g.connections());
      stato++;
      f.setEnabled(false);
      break;
    }
  }
}